package ru.yandex.webmaster3.viewer.http.checklist.extras;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.TurboUrlErrorSample;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistPageSamplesService;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistSamplesType;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * Created by Oleg Bazdyrev on 2019-09-18.
 */
@ReadAction
@Description("Получение примеров ошибок урлов в турбо-фидах")
@Category("checklist")
@Component("/checklist/extras/turboUrlsErrorSamples")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TurboUrlsErrorSamplesAction extends AbstractUserVerifiedHostAction<PageSamplesRequest, TurboUrlsErrorSamplesAction.Response> {

    private static final ChecklistSamplesType SAMPLES_TYPE = ChecklistSamplesType.TURBO_URL_ERRORS;

    private final SiteProblemsService siteProblemsService;
    private final ChecklistPageSamplesService checklistPageSamplesService;


    @Override
    public Response process(PageSamplesRequest request) {
        AbstractProblemInfo problemInfo = siteProblemsService.getRealTimeProblemInfo(request.getHostId(), SAMPLES_TYPE.getProblemType());
        if (problemInfo == null || problemInfo.getState() == SiteProblemState.ABSENT) {
            return new TurboUrlsErrorSamplesAction.Response(Collections.emptyList(), 0);
        }
        PageUtils.Pager pager = PageUtils.getPage(request.getP(), request.getPSize());
        List<String> samples = checklistPageSamplesService.getSamples(request.getHostId(), SAMPLES_TYPE);
        List<TurboUrlErrorSample> results = samples.stream()
                .skip(pager.toRangeStart())
                .limit(pager.getPageSize())
                .map(str -> JsonMapping.readValue(str, TurboUrlErrorSample.class))
                .collect(Collectors.toList());
        return new Response(results, samples.size());
    }

    @Getter
    @RequiredArgsConstructor
    public class Response implements ActionResponse.NormalResponse {
        @Description("Примеры ошибок")
        private final List<TurboUrlErrorSample> samples;
        @Description("Всего примеров")
        private final int totalSamples;
    }
}
