package ru.yandex.webmaster3.viewer.http.checklist.extras;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistPageSamplesService;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistSamplesType;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.searchurl.samples.data.BadUrlSample;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author: ishalaru
 * DATE: 15.07.2019
 */
@Slf4j
@ReadAction
@Description("Получить примеры для алерта 4xx 5xx")
@Category("checklist")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class UrlAlertExtendedSamplesAction extends AbstractUserVerifiedHostAction<PageSamplesRequest, UrlAlertExtendedSamplesAction.UrlAlertSamplesResponse> {
    private final SiteProblemsService siteProblemsService;
    private final ChecklistPageSamplesService checklistPageSamplesService;
    @Setter
    private SiteProblemTypeEnum problemType;

    @Override
    public UrlAlertSamplesResponse process(PageSamplesRequest request) {
        AbstractProblemInfo problemInfo = siteProblemsService.getRealTimeProblemInfo(request.getHostId(), problemType);
        if (problemInfo == null || problemInfo.getContent() == null || problemInfo.getState() == problemType.getDefaultState()) {
            return new UrlAlertSamplesResponse(Collections.emptyList(), 0);
        }

        PageUtils.Pager pager = PageUtils.getPage(request.getP(), request.getPSize());
        final ChecklistSamplesType checklistSamplesType = ChecklistSamplesType.byProblemType(problemType);
        List<String> workValues = checklistPageSamplesService.getSamples(request.getHostId(), checklistSamplesType);
        List<BadUrlSample> samples = new ArrayList<>(workValues.size());

        for (String item : workValues) {
            samples.add(JsonMapping.readValue(item, BadUrlSample.class));
        }

        List<BadUrlSample> banSamples = samples.stream()
                .filter(Objects::nonNull)
                .sorted(Comparator.comparing(BadUrlSample::getLastAccess, Comparator.reverseOrder()))
                .skip(pager.toRangeStart())
                .limit(pager.getPageSize())
                .collect(Collectors.toList());
        return new UrlAlertSamplesResponse(banSamples, samples.size());
    }

    @AllArgsConstructor
    @Getter
    public static class UrlAlertSamplesResponse implements ActionResponse.NormalResponse {
        private final List<BadUrlSample> samples;
        private final int totalSamples;
    }
}
