package ru.yandex.webmaster3.viewer.http.checklist.extras;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.checklist.data.YaBrowserBadAdSample;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistPageSamplesService;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistSamplesType;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.checklist.extras.YaBrowserBadAdSamplesAction.Response;

/**
 * @author avhaliullin
 */
@ReadAction
@Description("Получить примеры проблемных страниц с плохой рекламой")
@Category("checklist")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class YaBrowserBadAdSamplesAction extends AbstractUserVerifiedHostAction<PageSamplesRequest, Response> {

    private final SiteProblemsService siteProblemsService;
    private final ChecklistPageSamplesService checklistPageSamplesService;


    @Override
    public Response process(PageSamplesRequest request) {
        AbstractProblemInfo problemInfo = siteProblemsService.getRealTimeProblemInfo(
                request.getHostId(), SiteProblemTypeEnum.YABROWSER_BADAD);
        if (problemInfo == null || problemInfo.getState() == SiteProblemState.ABSENT) {
            return new Response(Collections.emptyList(), 0);
        }
        PageUtils.Pager pager = PageUtils.getPage(request.getP(), request.getPSize());
        List<String> samples = checklistPageSamplesService.getSamples(request.getHostId(), ChecklistSamplesType.YABROWSER_BADAD);
        List<YaBrowserBadAdSample> results = samples.stream()
                .skip(pager.toRangeStart())
                .limit(pager.getPageSize())
                .map(str -> {
                    try {
                        return JsonDBMapping.OM.readValue(str, YaBrowserBadAdSample.class);
                    } catch (IOException e) {
                        // fallback
                        return new YaBrowserBadAdSample(null, IdUtils.hostIdToUrl(request.getHostId()) + str);
                    }
                }).collect(Collectors.toList());
        return new Response(results, samples.size());
    }

    public static final class Response implements ActionResponse.NormalResponse {

        private final List<YaBrowserBadAdSample> samples;
        private final int totalSamples;

        public Response(List<YaBrowserBadAdSample> samples, int totalSamples) {
            this.samples = samples;
            this.totalSamples = totalSamples;
        }

        public List<YaBrowserBadAdSample> getSamples() {
            return samples;
        }

        public int getTotalSamples() {
            return totalSamples;
        }
    }
}
