package ru.yandex.webmaster3.viewer.http.comment;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.Data;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.comment.CmntIntegraionService;
import ru.yandex.webmaster3.core.comment.model.LazyInitChatData;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.review.ComplaintService;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserRequest;

/**
 * author: ishalaru
 * DATE: 22.07.2019
 */
@ReadAction
@Description(value = "Получить дополнительную информацию по чату")
@Category("comment")
@Component("/comment/extendedLazyInfo")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Slf4j
public class GetCommentLazyExtendedAttribute extends
        AbstractUserAction<GetCommentLazyExtendedAttribute.Request, GetCommentLazyExtendedAttribute.Response> {

    private final ComplaintService complaintService;
    private final UserHostsService userHostsService;
    private final CmntIntegraionService cmntIntegraionService;

    @Override
    public Response process(Request request) {
        if (request.getHostId() == null && request.getSite() == null) {
            return new Response.UnDefinedHostId();
        }
        WebmasterHostId hostId = request.getHostId();
        if (hostId == null) {
            hostId = IdUtils.urlToHostId(request.getSite());
        }
        final List<UserVerifiedHost> verifiedHosts = userHostsService.getVerifiedHosts(request.getWebmasterUser(), List.of(hostId));
        if (verifiedHosts == null || verifiedHosts.isEmpty()) {
            return new Response.NotVerifiedHost();
        }

        List<String> reviewIds = request.infos.stream().map(EntityInfo::getEntityId).distinct().collect(Collectors.toList());
        Map<String, Boolean> abusedMap = getAbusedProperty(hostId, reviewIds);
        Map<String, String> chatTokens = getTokens(request.getUserId(), reviewIds);
        //cmntIntegraionService.initChats(request.infos.stream().map(e -> e.entityId).collect(Collectors.toList()));
        Map<String, String> lazyChatIds = cmntIntegraionService.lazyInitChats(createLazyChatData(request.getUserId(), request.infos));
        final Map<String, Response.ResponseInfoData> collect =
                reviewIds.stream()
                        .collect(Collectors.toMap(e -> e, e -> new Response.ResponseInfoData(abusedMap.get(e), chatTokens.get(e), lazyChatIds.get(e))));
        //Проставить признаки того что данные отзыв, нужно отослать на ревью. И софрмировать строку для официльного отзыва
        return new Response.NormalResponseImpl(collect);
    }

    public LazyInitChatData createLazyChatData(Long userId, List<EntityInfo> entityIds) {
        List<LazyInitChatData.Entity> list = new ArrayList<>();
        for (EntityInfo info : entityIds) {
            LazyInitChatData.Query query = new LazyInitChatData.Query("initChatData", info.getEntityId(),
                    info.getSupplierServiceSlug(), info.getName(), info.getVersion());
            LazyInitChatData.Entity entity = new LazyInitChatData.Entity(info.getEntityId(), List.of(query));
            list.add(entity);
        }
        return new LazyInitChatData(userId, list);
    }


    private Map<String, Boolean> getAbusedProperty(WebmasterHostId hostId, Collection<String> reviewIds) {
        final Set<String> check = new HashSet<>(complaintService.check(hostId, reviewIds));
        return reviewIds.stream().collect(Collectors.toMap(e -> e, check::contains, W3Collectors.throwingMerger()));
    }

    private Map<String, String> getTokens(Long userId, Collection<String> chatIds) {
        return cmntIntegraionService.chatsSignInit(userId, chatIds);
    }

    @Getter
    @Setter(onMethod_ = {@RequestQueryProperty})
    public static class Request extends AbstractUserRequest {
        @Description("")
        private List<EntityInfo> infos;

        @Description("hostId для которого беруться данные")
        private WebmasterHostId hostId;

        @Description("Cайт для которого беруться данные")
        private String site;
    }

    @Data
    public static class EntityInfo {
        @Description("ID отзыва, оно же ID чата")
        String entityId;
        @Description("Текст review")
        String name;
        @Description("Сервис (UGC)")
        String supplierServiceSlug;
        @Description("Версия")
        Long version;
    }

    public interface Response extends ActionResponse {
        @Value
        class NormalResponseImpl implements Response, NormalResponse {
            @Description("Мапа ответов")
            Map<String, ResponseInfoData> reviewInfo;
        }

        class UnDefinedHostId implements Response, ErrorResponse {

            @Override
            public Enum<?> getCode() {
                return ErrorCode.UNDEFINED_HOST_ID;
            }

            @Override
            public String getMessage() {
                return "Site or hostId must be defined.";
            }
        }

        class NotVerifiedHost implements Response, ErrorResponse {

            @Override
            public Enum<?> getCode() {
                return ErrorCode.HOST_IS_NOT_VERIFIED;
            }

            @Override
            public String getMessage() {
                return "Site or hostId is not verified for user.";
            }
        }


        enum ErrorCode {
            UNDEFINED_HOST_ID, HOST_IS_NOT_VERIFIED,
        }

        @Value
        class ResponseInfoData {
            @Description("Признак того что на отзыв жаловались")
            boolean reviewAbused;
            @Description("Security токен как признак официальности, для ответа")
            String token;
            @Description("Security токен для линивой инициализации чата")
            String lazyInitToken;
        }

    }

}

