package ru.yandex.webmaster3.viewer.http.crawl;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.url.checker3.dao.UrlCheckDataBlocksYDao;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlock;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockState;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockType;
import ru.yandex.webmaster3.storage.url.checker3.service.UrlCheckDataBlocksService;
import ru.yandex.webmaster3.viewer.http.BaseWebmaster3Action;
import ru.yandex.webmaster3.viewer.http.async.model.AsyncInfoRequest;
import ru.yandex.webmaster3.viewer.http.crawl.data.UrlRotorCheckInfo;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author leonidrom
 */
@Description("Получить результат проверки роторения урла")
@Slf4j
@ReadAction
@Category("crawl")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/host/rotorCheck/info")
public class GetUrlRotorCheckInfoAction extends BaseWebmaster3Action<AsyncInfoRequest, GetUrlRotorCheckInfoResponse> {
    private final UrlCheckDataBlocksService urlCheckDataBlocksService;

    @Override
    public GetUrlRotorCheckInfoResponse process(AsyncInfoRequest request) {
        var dataBlocks = urlCheckDataBlocksService.getWithoutData(request.getRequestId());
        if (dataBlocks.isEmpty()) {
            return new GetUrlRotorCheckInfoResponse.RequestIdNotFoundResponse();
        }

        log.info("Data blocks: {}", Arrays.toString(dataBlocks.toArray()));

        if (UrlCheckDataBlock.anyOfState(dataBlocks, UrlCheckDataBlockState.INTERNAL_ERROR)) {
            return new GetUrlRotorCheckInfoResponse.InternalErrorResponse();
        }

        if (UrlCheckDataBlock.anyOfState(dataBlocks, UrlCheckDataBlockState.ROBOT_FETCH_ERROR)) {
            return new GetUrlRotorCheckInfoResponse.UnableToDownloadContentResponse();
        }

        if (UrlCheckDataBlock.anyOfState(dataBlocks, UrlCheckDataBlockState.TIMED_OUT)) {
            return new GetUrlRotorCheckInfoResponse.TimedOutResponse();
        }

        if (UrlCheckDataBlock.anyOfState(dataBlocks, UrlCheckDataBlockState.IN_PROGRESS)) {
            return new GetUrlRotorCheckInfoResponse.NormalResponse();
        }

        dataBlocks = urlCheckDataBlocksService.get(request.getRequestId());
        Map<UrlCheckDataBlockType, UrlCheckDataBlock> dataBlocksMap = dataBlocks.stream()
                .collect(Collectors.toMap(UrlCheckDataBlock::getBlockType, b -> b));

        UrlRotorCheckInfo checkInfoJSEnabled = UrlRotorCheckInfo.fromDataBlocks(
                dataBlocksMap.get(UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_ON),
                dataBlocksMap.get(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_ON),
                dataBlocksMap.get(UrlCheckDataBlockType.ROBOT_SERVER_RESPONSE_RENDER_ON)
        );

        UrlRotorCheckInfo checkInfoJSDisabled = UrlRotorCheckInfo.fromDataBlocks(
                dataBlocksMap.get(UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_OFF),
                dataBlocksMap.get(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_OFF),
                dataBlocksMap.get(UrlCheckDataBlockType.ROBOT_SERVER_RESPONSE_RENDER_OFF)
        );

        return new GetUrlRotorCheckInfoResponse.NormalResponse(checkInfoJSEnabled, checkInfoJSDisabled);
    }
}
