package ru.yandex.webmaster3.viewer.http.delurl;

import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.delurl.DelUrlRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.request.PagerAware;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.storage.delurl.DelUrlRequestsService;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

import java.util.Collections;
import java.util.List;

import static ru.yandex.webmaster3.viewer.http.delurl.GetDelurlRequestsAction.Request;
import static ru.yandex.webmaster3.viewer.http.delurl.GetDelurlRequestsAction.Response;

/**
 * Created by Oleg Bazdyrev on 10/10/2017.
 */
@ReadAction
@Category("delurl")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class GetDelurlRequestsAction extends AbstractUserVerifiedHostAction<Request, Response> {
    private final DelUrlRequestsService delurlRequestsService;

    @Override
    public Response process(Request request) {
        try {
            int count = delurlRequestsService.count(request.getHostId(), request.getFromDate(), request.getToDate());
            PageUtils.Pager page = PageUtils.getPage(request.getPage(), request.getPageSize(), count);
            if (page.isEmpty()) {
                return new Response(Collections.emptyList(), count);
            }

            List<DelUrlRequest> reqs = delurlRequestsService.list(request.getHostId(), request.getFromDate(),
                    request.getToDate(), page.toRangeStart(), page.getPageSize());

            return new Response(reqs, count);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Unable to read delurl requests from Ydb",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public static class Request extends AbstractUserVerifiedHostRequest implements PagerAware {
        private int page = 0;
        private int pageSize = 25;
        private DateTime fromDate;
        private DateTime toDate;

        @Override
        public void setP(int page) {
            this.page = page;
        }

        @Override
        public void setPSize(int pageSize) {
            this.pageSize = pageSize;
        }

        public int getPage() {
            return page;
        }

        public int getPageSize() {
            return pageSize;
        }

        public DateTime getFromDate() {
            return fromDate;
        }

        @RequestQueryProperty
        public void setFromDate(DateTime fromDate) {
            this.fromDate = fromDate;
        }

        public DateTime getToDate() {
            return toDate;
        }

        @RequestQueryProperty
        public void setToDate(DateTime toDate) {
            this.toDate = toDate;
        }
    }

    public static final class Response implements ActionResponse.NormalResponse {

        private final List<DelUrlRequest> requests;
        private final int totalRequests;

        public Response(List<DelUrlRequest> requests, int totalRequests) {
            this.requests = requests;
            this.totalRequests = totalRequests;
        }

        public List<DelUrlRequest> getRequests() {
            return requests;
        }

        public int getTotalRequests() {
            return totalRequests;
        }
    }
}
