package ru.yandex.webmaster3.viewer.http.download.urls;

import com.google.common.base.Charsets;
import com.google.common.hash.Funnel;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskData;
import ru.yandex.webmaster3.storage.download.urls.DownloadEventUrlsTaskData;
import ru.yandex.webmaster3.storage.searchurl.SearchUrlEventFilter;
import ru.yandex.webmaster3.storage.searchurl.SearchUrlSamplesService;
import ru.yandex.webmaster3.viewer.http.download.DownloadAction;

/**
 * Created by ifilippov5 on 13.02.17.
 */
@ReadAction
@Description("Выгрузка событий")
@Category("searchurl")
public class DownloadEventUrlsAction extends DownloadAction<DownloadEventUrlsRequest> {
    private SearchUrlSamplesService searchUrlSamplesService;

    @Override
    protected WorkerTaskData getTaskData(DownloadEventUrlsRequest request, long hash) {
        String fileName = generateFileName(request, hash);
        return DownloadEventUrlsTaskData.createTaskData(
                request.getHostId(),
                request.getParameterFilters(),
                request.getNodeId(),
                request.getExportFormat(),
                fileName,
                hash
        );
    }

    @Override
    protected long getHash(DownloadEventUrlsRequest request) {
        HashFunction hf = Hashing.md5();

        Hasher hc = hf.newHasher()
                .putInt(request.getHostId().hashCode())
                .putString(request.getExportFormat().toString(), Charsets.UTF_8)
                .putString(searchUrlSamplesService.getEventSamplesDataVersion(request.getHostId()), Charsets.UTF_8);

        if (request.getNodeId() != null)
            hc.putLong(request.getNodeId());

        if (request.getParameterFilters() != null) {
            for (SearchUrlEventFilter filter : request.getParameterFilters()) {
                hc.putObject(filter, (Funnel<SearchUrlEventFilter>) (searchUrlFilter, into) -> {
                    into.putString(searchUrlFilter.getOperation().toString(), Charsets.UTF_8);
                    into.putString(searchUrlFilter.getIndicator().toString(), Charsets.UTF_8);
                    into.putString(searchUrlFilter.getValue(), Charsets.UTF_8);
                });
            }
        }

        return hc.hash().asLong();
    }

    @Required
    public void setSearchUrlSamplesService(SearchUrlSamplesService searchUrlSamplesService) {
        this.searchUrlSamplesService = searchUrlSamplesService;
    }
}

