package ru.yandex.webmaster3.viewer.http.feedback;

import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;
import java.util.TreeSet;

import lombok.extern.slf4j.Slf4j;
import lombok.extern.slf4j.XSlf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.feedback.FeedbackType;
import ru.yandex.webmaster3.core.feedback.SendFeedbackEMailTaskData;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.storage.feedback.dao.FeedbackYDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserAction;
import ru.yandex.webmaster3.viewer.http.feedback.model.FeedbackCreateRequest;
import ru.yandex.webmaster3.viewer.http.feedback.model.FeedbackCreateResponse;

import static ru.yandex.webmaster3.core.feedback.FeedbackType.NOTIFICATIONS_OFF;

/**
 * @author: ishalaru
 * DATE: 03.06.2019
 */
@WriteAction
@Category("feedback")
@Description(value = "Создание нового фидбека")
@Component("/feedback/add")
@Slf4j
public class FeedbackCreateAction extends AbstractUserAction<FeedbackCreateRequest,
        FeedbackCreateResponse> {
    private static final int MAX_MESSAGE_LENGTH = 1024;
    private static final int MAX_META_LENGTH = 2048;

    private final FeedbackYDao feedbackYDao;
    private final WorkerClient workerClient;

    @Autowired
    public FeedbackCreateAction(FeedbackYDao feedbackYDao,

                                @Qualifier("lbWorkerClient") WorkerClient workerClient) {
        this.feedbackYDao = feedbackYDao;
        this.workerClient = workerClient;
    }

    private static final EnumSet<FeedbackType> TYPES_WITHOUT_HOST = EnumSet.of(NOTIFICATIONS_OFF);

    @Override
    public FeedbackCreateResponse process(FeedbackCreateRequest request) {
        if (request.getMessage() == null || request.getMessage().isBlank()) {
            return new FeedbackCreateResponse.MessageIsEmpty();
        } else if (request.getMessage().trim().length() > MAX_MESSAGE_LENGTH) {
            return new FeedbackCreateResponse.MessageIsTooLong();
        }
        if (request.getMeta() != null && request.getMeta().length() > MAX_META_LENGTH) {
            return new FeedbackCreateResponse.MetaIsTooLong();
        }
        if (!TYPES_WITHOUT_HOST.contains(request.getType()) && request.getHostId() == null) {
            return new FeedbackCreateResponse.HostIdIsMissing();
        }

        feedbackYDao.insert(request.getHostId(), request.getUserId(), DateTime.now(),
                request.getType(), request.getMessage().trim(), request.getMeta());

        var task = new SendFeedbackEMailTaskData(request.getHostId(), request.getUserId(),
                request.getMessage(), request.getMeta(), request.getType());
        log.debug(task.toString());
        workerClient.enqueueTask(task);

        return new FeedbackCreateResponse.NormalResponse();
    }
}

