package ru.yandex.webmaster3.viewer.http.feeds;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import org.apache.commons.lang3.tuple.Triple;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.FeedsErrorSeverity;
import ru.yandex.webmaster3.core.feeds.feed.FeedsQualityErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.FeedsValidationErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao.FeedRecord;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.SerpdataLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.models.FeedStats;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.feeds.GetFeeds2Action.ExtendedFeedInfo;

import static ru.yandex.webmaster3.core.feeds.feed.NativeFeedType.STORES;


/**
 * @author kravchenko99
 * @date 7/30/21
 */

@Component("/feeds/getNativeFeedInfo")
@ReadAction
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Category("feeds")
public class GetFeedInfoAction extends Action<GetFeedInfoAction.Req, GetFeedInfoAction.Res> {

    private final FeedsNative2YDao feedsNativeYDao;
    private final FeedsOffersLogsHistoryCHDao feedsOffersLogsHistoryCHDao;
    private final FeedsService feedsService;
    private final GoodsOffersLogsHistoryCHDao goodsOffersLogsHistoryCHDao;
    private final SerpdataLogsHistoryCHDao serpdataLogsHistoryCHDao;

    @Override
    public Res process(Req request) throws WebmasterException {
        WebmasterHostId hostId = request.getHostId();
        String domain = WwwUtil.cutWWWAndM(hostId);
        // temp parse url + login + password
        FeedsService.UrlWithUserInfo urlWithUserInfo = feedsService.parseUrl(request.url);
        request.url = urlWithUserInfo.url();

        NativeFeedInfo2 info = feedsNativeYDao.get(domain, request.url);
        info = info.toUrlWithUserInfo();
        if (info.getStatusScc() == NativeFeedSccStatus.SUCCESS) {
            info = info.withErrorsScc(null);
        }

        FeedRecord offerState;
        if (info.getType() == NativeFeedType.STORES) {
            offerState = goodsOffersLogsHistoryCHDao.getLastState(Collections.singleton(Triple.of(info.getBusinessId(), info.getPartnerId(), info.getFeedId())))
                    .stream().map(GoodsOffersLogsHistoryCHDao.GoodsFeedRecord::toFeedRecord).findFirst().orElse(null);
        } else {
            offerState = Iterables.getFirst(feedsOffersLogsHistoryCHDao.getLastState(Collections.singleton(request.url)), null);
        }

        FeedRecord serpdataState = Iterables.getFirst(serpdataLogsHistoryCHDao.getLastState(domain, null, null), null);
        FeedStats offerStats = Optional.ofNullable(offerState).map(FeedRecord::getStats).orElse(FeedStats.EMPTY);
        FeedStats serpdataStats = Optional.ofNullable(serpdataState).map(FeedRecord::getStats).orElse(FeedStats.EMPTY);

        NativeFeedStatus status = feedsService.getStatus(info.getType(), offerState, serpdataState);

        var builder = ExtendedFeedInfo.builder()
                .domain(info.getDomain())
                .url(info.getUrl())
                .userId(info.getUserId())
                .regionsId(info.getRegionsId())
                .login(info.getLogin())
                .password(info.getPassword())
                .status(status)
                .statusVertical(info.getStatusVertical())
                .errors(info.getErrors())
                .statusScc(info.getStatusScc())
                .errorsScc(info.getErrorsScc())
                .type(info.getType())
                .enabledServiceTypes(info.getEnabledServiceTypes())
                .addDate(info.getAddDate())
                .updateDate(offerState == null ? info.getUpdateDate() : offerState.getLastAccess())
                .enabled(info.isEnabled())
                .businessId(info.getBusinessId())
                .partnerId(info.getPartnerId())
                .feedId(info.getFeedId())
                .sccTimestamp(info.getSccTimestamp())
                .errorsOfferBase(info.getErrorsOfferBase())
                .errorOffers(offerStats.getError() + serpdataStats.getError())
                .totalOffers(offerStats.getTotal())
                .validOffers(offerStats.getOk())
                .warningOffers(offerStats.getWarning());

        if (offerState != null) {
            builder.validationErrorTypes(offerState.getErrors().entrySet().stream()
                    .filter(entry -> entry.getValue().stream().anyMatch(e -> e.getSeverity() == FeedsErrorSeverity.FATAL || e.getSeverity() == null))
                    .map(Map.Entry::getKey)
                    .map(FeedsValidationErrorEnum::byCode)
                    .filter(e -> e != FeedsValidationErrorEnum.Unknown).collect(Collectors.toSet()));
            builder.validationWarningTypes(offerState.getErrors().entrySet().stream()
                    .filter(entry -> entry.getValue().stream().anyMatch(e -> e.getSeverity() == FeedsErrorSeverity.WARNING))
                    .map(Map.Entry::getKey)
                    .map(FeedsValidationErrorEnum::byCode)
                    .filter(e -> e != FeedsValidationErrorEnum.Unknown).collect(Collectors.toSet()));
        } else {
            builder.validationErrorTypes(Collections.emptySet());
            builder.validationWarningTypes(Collections.emptySet());
        }
        if (serpdataState != null) {
            builder.qualityWarningTypes(serpdataState.getErrors().keySet().stream().map(FeedsQualityErrorEnum::byCode)
                    .filter(e -> e != FeedsQualityErrorEnum.Unknown).collect(Collectors.toSet()));
        } else {
            builder.qualityWarningTypes(Collections.emptySet());
        }

        ExtendedFeedInfo extendedFeedInfo = builder.build();

        return new Res(info, info, extendedFeedInfo);
    }

    public static class Req extends AbstractUserVerifiedHostRequest {
        @Setter(onMethod_ = @RequestQueryProperty)
        NativeFeedType type = STORES;
        @Setter(onMethod_ = @RequestQueryProperty(required = true))
        String url;
        @Setter(onMethod_ = @RequestQueryProperty)
        int regionId = 0;
    }

    @Value
    public static class Res implements ActionResponse.NormalResponse {
        NativeFeedInfo2 feedInfo;
        NativeFeedInfo2 newFeedInfo;
        ExtendedFeedInfo extendedFeedInfo;
    }

}
