package ru.yandex.webmaster3.viewer.http.feeds;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.Builder;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Triple;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.FeedServiceType;
import ru.yandex.webmaster3.core.feeds.feed.FeedsErrorSeverity;
import ru.yandex.webmaster3.core.feeds.feed.FeedsQualityErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.FeedsValidationErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao.FeedRecord;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.SerpdataLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.models.FeedStats;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;


/**
 * @author kravchenko99
 * @date 7/30/21
 */

@Component("/feeds/getNativeFeeds2")
@ReadAction
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Category("feeds")
@Slf4j
public class GetFeeds2Action extends Action<GetFeeds2Action.Req, GetFeeds2Action.Res> {

    private final FeedsService feedsService;
    private final FeedsNative2YDao feedsNativeYDao;
    private final FeedsOffersLogsHistoryCHDao feedsOffersLogsHistoryCHDao;
    private final GoodsOffersLogsHistoryCHDao goodsOffersLogsHistoryCHDao;
    private final SerpdataLogsHistoryCHDao serpdataLogsHistoryCHDao;

    @Override
    public Res process(Req request) throws WebmasterException {
        WebmasterHostId hostId = request.getHostId();
        String domain = WwwUtil.cutWWWAndM(hostId);
        List<NativeFeedInfo2> rawFeedInfos = feedsNativeYDao.list(domain);
        Map<String, FeedRecord> offersStateByUrl = feedsOffersLogsHistoryCHDao.getLastState(
                        rawFeedInfos.stream().filter(NativeFeedInfo2::isNotGoodsFeed)
                                .map(NativeFeedInfo2::getUrl).collect(Collectors.toList())
                ).stream().filter(Objects::nonNull).collect(Collectors.toMap(FeedRecord::getUrl, Function.identity()));
        Map<String, FeedRecord> serpdataStateByUrl = serpdataLogsHistoryCHDao.getLastState(domain, null, null)
                .stream().filter(Objects::nonNull).collect(Collectors.toMap(FeedRecord::getUrl, Function.identity()));

        var feedIds = rawFeedInfos.stream().filter(NativeFeedInfo2::isGoodsFeed)
                .filter(f -> f.getBusinessId() != null && f.getPartnerId() != null && f.getFeedId() != null)
                .map(f -> Triple.of(f.getBusinessId(), f.getPartnerId(), f.getFeedId()))
                .collect(Collectors.toList());
        goodsOffersLogsHistoryCHDao.getLastState(feedIds).forEach(goodsFeedRecord -> {
            offersStateByUrl.put(goodsFeedRecord.getFeedUrl(), goodsFeedRecord.toFeedRecord());
        });

        List<ExtendedFeedInfo> result = new ArrayList<>(rawFeedInfos.size());
        for (NativeFeedInfo2 rawFeed : rawFeedInfos) {
            FeedRecord offerState = offersStateByUrl.get(rawFeed.getUrl());
            FeedRecord serpdataState = serpdataStateByUrl.get(rawFeed.getUrl());
            FeedStats errorStats = Optional.ofNullable(offerState).map(FeedRecord::getErrorStats).orElse(FeedStats.EMPTY);
            FeedStats serpdataStats = Optional.ofNullable(serpdataState).map(FeedRecord::getErrorStats).orElse(FeedStats.EMPTY);
            // change status
            long errorCount = errorStats.getError() + serpdataStats.getError();
            // status calc (non-stores)
            NativeFeedStatus status = feedsService.getStatus(rawFeed.getType(), offerState, serpdataState);

            DateTime lastAccess = offerState == null ? rawFeed.getUpdateDate() : offerState.getLastAccess();
            rawFeed = rawFeed.toUrlWithUserInfo();
            // TODO можно ли это как-то упростить?
            final ExtendedFeedInfo.ExtendedFeedInfoBuilder builder = ExtendedFeedInfo.builder();
            builder
                    .domain(rawFeed.getDomain())
                    .url(rawFeed.getUrl())
                    .userId(rawFeed.getUserId())
                    .regionsId(rawFeed.getRegionsId())
                    .login(rawFeed.getLogin())
                    .password(rawFeed.getPassword())
                    .status(status)
                    .statusVertical(rawFeed.getStatusVertical())
                    .errors(rawFeed.getErrors())
                    .statusScc(rawFeed.getStatusScc())
                    .errorsScc(rawFeed.getErrorsScc())
                    .type(rawFeed.getType())
                    .enabledServiceTypes(rawFeed.getEnabledServiceTypes())
                    .addDate(rawFeed.getAddDate())
                    .updateDate(lastAccess)
                    .enabled(rawFeed.isEnabled())
                    .businessId(rawFeed.getBusinessId())
                    .partnerId(rawFeed.getPartnerId())
                    .feedId(rawFeed.getFeedId())
                    .sccTimestamp(rawFeed.getSccTimestamp())
                    .errorsOfferBase(rawFeed.getErrorsOfferBase())
                    .errorOffers(errorCount)
                    .totalOffers(errorStats.getTotal())
                    .validOffers(errorStats.getOk())
                    .warningOffers(errorStats.getWarning());
            if (offerState != null) {
                builder.validationErrorTypes(offerState.getErrors().entrySet().stream()
                        .filter(entry -> entry.getValue().stream().anyMatch(e -> e.getSeverity() == FeedsErrorSeverity.FATAL || e.getSeverity() == null))
                        .map(Map.Entry::getKey)
                        .map(FeedsValidationErrorEnum::byCode)
                        .filter(e -> e != FeedsValidationErrorEnum.Unknown).collect(Collectors.toSet()));
                builder.validationWarningTypes(offerState.getErrors().entrySet().stream()
                        .filter(entry -> entry.getValue().stream().anyMatch(e -> e.getSeverity() == FeedsErrorSeverity.WARNING))
                        .map(Map.Entry::getKey)
                        .map(FeedsValidationErrorEnum::byCode)
                        .filter(e -> e != FeedsValidationErrorEnum.Unknown).collect(Collectors.toSet()));
            } else {
                builder.validationErrorTypes(Collections.emptySet());
                builder.validationWarningTypes(Collections.emptySet());
            }
            if (serpdataState != null) {
                builder.qualityWarningTypes(serpdataState.getErrors().keySet().stream().map(FeedsQualityErrorEnum::byCode)
                        .filter(e -> e != FeedsQualityErrorEnum.Unknown).collect(Collectors.toSet()));
            } else {
                builder.qualityWarningTypes(Collections.emptySet());
            }

            result.add(builder.build());
        }
        return new Res(result, result);
    }

    @Setter(onMethod_ = @RequestQueryProperty)
    public static class Req extends AbstractUserVerifiedHostRequest {
    }

    @Value
    public static class Res implements ActionResponse.NormalResponse {
        List<ExtendedFeedInfo> feedInfos;

        List<ExtendedFeedInfo> feedInfos2;
    }

    @Value
    @Builder
    public static class ExtendedFeedInfo {
        String domain;
        String url;
        Long userId;
        List<Integer> regionsId;
        String login;
        String password;
        NativeFeedStatus status;
        NativeFeedStatus statusVertical;
        List<String> errors;
        NativeFeedSccStatus statusScc;
        List<String> errorsScc;
        NativeFeedType type;
        Set<FeedServiceType> enabledServiceTypes;
        DateTime addDate;
        DateTime updateDate;
        boolean enabled;
        Long businessId;
        Long partnerId;
        Long feedId;
        DateTime sccTimestamp;
        List<Map<String, Object>> errorsOfferBase;
        @Description("Всего офферов в прокачке")
        Long totalOffers;
        @Description("Валидных предложений в прокачке")
        Long validOffers;
        Long errorOffers;
        Long warningOffers;
        Set<FeedsValidationErrorEnum> validationErrorTypes;
        Set<FeedsValidationErrorEnum> validationWarningTypes;
        Set<FeedsQualityErrorEnum> qualityWarningTypes;
    }

}
