package ru.yandex.webmaster3.viewer.http.feeds;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.FeedsDefectRateErrorInfo;
import ru.yandex.webmaster3.core.feeds.feed.FeedsErrorSeverity;
import ru.yandex.webmaster3.core.feeds.feed.FeedsQualityErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsDefectRateErrorYDao;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.SerpdataLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.models.FeedStats;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;


/**
 * @author kravchenko99
 * @date 7/30/21
 */

@Component("/feeds/getDomainError")
@ReadAction
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Category("feeds")
public class GetFeedsDomainErrorAction extends Action<GetFeedsDomainErrorAction.Req, GetFeedsDomainErrorAction.Res> {

    private static final String EMPTY_FEED = "";

    private final FeedsDefectRateErrorYDao feedsDomainErrorYDao;
    private final SerpdataLogsHistoryCHDao serpdataLogsHistoryCHDao;

    @Override
    public Res process(Req request) throws WebmasterException {
        WebmasterHostId hostId = request.getHostId();
        String domain = WwwUtil.cutWWWAndM(hostId);
        List<FeedsOffersLogsHistoryCHDao.FeedRecord> lastState = serpdataLogsHistoryCHDao.getLastState(domain, null,
                Collections.singleton(EMPTY_FEED));

        List<SerpDataDomainErrors> serpDataErrors = new ArrayList<>();

        for (FeedsOffersLogsHistoryCHDao.FeedRecord feedRecord : lastState) {
            NativeFeedType type = NativeFeedType.fromCode(feedRecord.getType());
            FeedStats serpdataStats = Objects.requireNonNullElse(feedRecord.getErrorStats(), FeedStats.EMPTY);
            List<FeedsQualityErrorEnum> serpDataErrorStatuses =
                    feedRecord.getErrors().keySet().stream().map(FeedsQualityErrorEnum::byCode).toList();

            serpDataErrors.add(new SerpDataDomainErrors(type, serpDataErrorStatuses,
                    serpdataStats.getError() > 0 || serpdataStats.getWarning() > 0 ? FeedsErrorSeverity.WARNING : FeedsErrorSeverity.SUCCESS));
        }

        return new Res(feedsDomainErrorYDao.list(domain), serpDataErrors);
    }

    @Setter(onMethod_ = @RequestQueryProperty(required = true))
    public static class Req extends AbstractUserVerifiedHostRequest {
    }

    @Value
    private static class SerpDataDomainErrors {
        NativeFeedType type;
        List<FeedsQualityErrorEnum> statuses;
        FeedsErrorSeverity severity;
    }

    @Value
    public static class Res implements ActionResponse.NormalResponse {
        List<FeedsDefectRateErrorInfo> defectRateErrors;
        List<SerpDataDomainErrors> serpDataErrors;
    }

}
