package ru.yandex.webmaster3.viewer.http.feeds;

import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.async.AsyncRequestTaskService;
import ru.yandex.webmaster3.storage.async.model.AsyncOperationRequestInfo;
import ru.yandex.webmaster3.storage.async.model.HttpAsyncRequestInfoRequestState;
import ru.yandex.webmaster3.storage.async.model.HttpAsyncRequestInfoResultType;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.robotstxt.model.AnalyzeRobotsTxtResponseType;
import ru.yandex.webmaster3.viewer.http.BaseWebmaster3Action;
import ru.yandex.webmaster3.viewer.http.async.model.AsyncInfoRequest;

/**
 * @author: kravchenko99
 * DATE: 21.11.2021
 */
@ReadAction
@Category("feeds")
@Slf4j
@Component("/feeds/addFeed/info")
public class InfoAddFeedAction extends BaseWebmaster3Action<AsyncInfoRequest, AddFeedResponse> {
    private static final Duration ANSWER_WAITING_TIMEOUT = Duration.standardSeconds(65);

    @Autowired
    private AsyncRequestTaskService asyncRequestTaskService;

    @Override
    public AddFeedResponse process(AsyncInfoRequest request) {
        AsyncOperationRequestInfo responseInfo = asyncRequestTaskService.getResponseInfo(request.getRequestId());
        if (responseInfo == null) {
            return new AddFeedResponse.RequestIdNotFoundResponse();
        }
        if (responseInfo.getState() == HttpAsyncRequestInfoRequestState.NEW &&
                responseInfo.getLastUpdate().plus(ANSWER_WAITING_TIMEOUT).isBefore(DateTime.now())) {
            log.error("Worker failed to finish request in {}", ANSWER_WAITING_TIMEOUT);
            return new AddFeedResponse.TimedOutResponse();
        }

        if (responseInfo.getState() != HttpAsyncRequestInfoRequestState.TASK_FINISHED) {
            return NormalResponseInfo.stillInProgress();
        }
        if (responseInfo.getResult() != HttpAsyncRequestInfoResultType.OK) {
            if (responseInfo.getResult() == HttpAsyncRequestInfoResultType.INTERNAL_ERROR) {
                throw new WebmasterException("InternalError", new WebmasterErrorResponse.AsyncZoraErrorResponse(getClass()));
            }
        }
        if (responseInfo.getResponseObjectTypeId() == AnalyzeRobotsTxtResponseType.NORMAL.getValue()) {
            FeedsService.ZoraCheckResult result = convertData(responseInfo, FeedsService.ZoraCheckResult.class);
            return switch (result) {
                case OK -> new NormalResponseInfo(false);
                case TIMED_OUT -> new AddFeedResponse.TimedOutResponse();
                case BAD_CODE -> new AddFeedResponse.BadUrlHttpCodeResponse();
                case BAD_MIME_TYPE -> new AddFeedResponse.BadUrlMimeTypeResponse();
                case BANNED -> new AddFeedResponse.FeedBannedResponse();
            };
        }
        throw new WebmasterException("InternalError", new WebmasterErrorResponse.AsyncAddFeedErrorResponse(getClass()));
    }

    protected <WorkClass> WorkClass convertData(AsyncOperationRequestInfo info, Class<WorkClass> clazz) {
        WorkClass result = null;
        try {
            result = JsonMapping.readValue(info.getResponse(), clazz);
        } catch (Exception e) {
            log.error("Problem with parsing response.", e);
        }
        return result;
    }

    @Value
    public static class NormalResponseInfo extends AddFeedResponse.NormalResponse {
        boolean inProgress;

        public static NormalResponseInfo stillInProgress() {
            return new NormalResponseInfo(true);
        }
    }
}
