package ru.yandex.webmaster3.viewer.http.feeds;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;


/**
 * @author kravchenko99
 * @date 7/30/21
 */

@Component("/feeds/removeFeed2")
@WriteAction
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Category("feeds")
public class RemoveFeed2Action extends Action<RemoveFeed2Action.Req, RemoveFeed2Action.Res> {
    private final FeedsService feedsService;


    @Override
    public Res process(Req request) throws WebmasterException {
        WebmasterHostId hostId = request.getHostId();
        String domain = WwwUtil.cutWWWAndM(hostId);

        var removeFeedStatus = feedsService.removeFeed(domain, request.url);
        return switch (removeFeedStatus) {
            case OK -> new Res.NormalResponse();
            case NOT_EXIST -> new Res.FeedAlreadyRemovedResponse();
        };
    }

    @Setter(onMethod_ = @RequestQueryProperty)
    public static class Req extends AbstractUserVerifiedHostRequest {
        private String url;
    }

    public abstract static class Res implements ActionResponse {

        public static class NormalResponse extends Res implements ActionResponse.NormalResponse {
        }


        public enum ErrorCode {
            FEED_ALREADY_REMOVED,
        }

        public abstract static class AbstractAddFeedError extends Res implements ErrorResponse {
            private final ErrorCode code;

            public AbstractAddFeedError(ErrorCode code) {
                this.code = code;
            }

            @Override
            public Enum<?> getCode() {
                return code;
            }
        }

        @Description("Такого фида с регионом нет")
        public static class FeedAlreadyRemovedResponse extends AbstractAddFeedError {
            public FeedAlreadyRemovedResponse() {
                super(ErrorCode.FEED_ALREADY_REMOVED);
            }
        }
    }
}
