package ru.yandex.webmaster3.viewer.http.feeds;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.annotation.Nullable;

import com.datastax.driver.core.utils.UUIDs;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.FeedServiceType;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.worker.task.feeds.ProcessingAddFeedTaskData;
import ru.yandex.webmaster3.storage.async.AsyncRequestTaskService;
import ru.yandex.webmaster3.storage.async.model.AsyncTaskType;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.feeds.models.ProcessingAddFeedInputData;
import ru.yandex.webmaster3.storage.turbo.dao.TurboFeedsSettingsYDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;


/**
 * @author kravchenko99
 * @date 7/30/21
 */

@Component("/feeds/addFeed/start")
@WriteAction
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Category("feeds")
public class StartAddFeedAction extends Action<StartAddFeedAction.Req, AddFeedResponse> {

    private final AsyncRequestTaskService asyncRequestTaskService;
    private final FeedsNative2YDao feedsNative2YDao;
    private final FeedsService feedsService;
    private final TurboFeedsSettingsYDao turboFeedsSettingsYDao;

    @Override
    public AddFeedResponse process(Req request) throws WebmasterException {
        WebmasterHostId hostId = request.getHostId();
        String domain = WwwUtil.cutWWWAndM(hostId);
        UUID taskId = UUIDs.timeBased();

        // костыль, решился не полагаться на фронт и сделал у нас, фиды типа Работа только для России
        if (request.type == NativeFeedType.VACANCY) {
            request.regionsId = List.of(FeedsService.RUSSIA);
        }

        if (CollectionUtils.isEmpty(request.regionsId)) {
            return new AddFeedResponse.NoRegionsResponse();
        }
        URL url = canonizeUrl(request.url);
        if (url == null) {
            return new AddFeedResponse.InvalidUrlResponse();
        }
        if (!url.getProtocol().equals("https")) {
            return new AddFeedResponse.HttpsOnlyResponse();
        }
        var urlWithUserInfo = feedsService.parseUrl(url);
        String urlStr = urlWithUserInfo.url();

        if ((urlWithUserInfo.login() != null || urlWithUserInfo.password() != null) &&
                (request.password != null || request.login != null)) {
            return new AddFeedResponse.InvalidUrlResponse();
        } else if (urlWithUserInfo.login() != null) {
            request.login = urlWithUserInfo.login();
            request.password = urlWithUserInfo.password();
        }

        if (feedsService.containsFeed(urlStr, request.type)) {
            return new AddFeedResponse.FeedAlreadyAddedResponse();
        }

        ProcessingAddFeedInputData processingAddFeedInputData = new ProcessingAddFeedInputData(hostId, urlStr,
                request.regionsId, request.getUserId(), request.type, request.enabledServiceTypes, request.login,
                request.password);

        ProcessingAddFeedTaskData taskData = new ProcessingAddFeedTaskData(taskId, hostId, urlStr,
                request.regionsId, request.getUserId(), request.type, request.enabledServiceTypes, request.login,
                request.password);


        AsyncRequestTaskService.CreateRequestResult requestResult =
                asyncRequestTaskService.createRequest(AsyncTaskType.ADD_FEED, 1, processingAddFeedInputData,
                        taskData);
        return new NormalResponse(requestResult.getRequestId());
    }

    @Nullable
    public static URL canonizeUrl(String requestUrl) {
        try {
            return new URL(requestUrl);
        } catch (MalformedURLException e) {
            return null;
        }
    }

    @Setter(onMethod_ = @RequestQueryProperty(required = true))
    public static class Req extends AbstractUserVerifiedHostRequest {
        String url;
        List<Integer> regionsId;
        NativeFeedType type;
        Set<FeedServiceType> enabledServiceTypes;
        String login;
        String password;

        @RequestQueryProperty
        public void setRegionsId(List<Integer> regionsId) {
            this.regionsId = regionsId;
        }

        @RequestQueryProperty
        public void setLogin(String login) {
            this.login = login;
        }

        @RequestQueryProperty
        public void setPassword(String password) {
            this.password = password;
        }

    }


    @Value
    public static class NormalResponse implements AddFeedResponse, ActionResponse.NormalResponse {
        UUID requestId;
    }
}
