package ru.yandex.webmaster3.viewer.http.feeds.logs;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.Builder;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.feeds.feed.FeedsQualityErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.FeedsValidationErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao.FeedRecord;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao.OfferErrorInfo;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao.GoodsFeedRecord;
import ru.yandex.webmaster3.storage.feeds.logs.SerpdataLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.models.FeedStats;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.feeds.logs.GetFeedsOffersLogsHistoryAction.Request;
import ru.yandex.webmaster3.viewer.http.feeds.logs.GetFeedsOffersLogsHistoryAction.Response;

/**
 * Created by Oleg Bazdyrev on 28/12/2021.
 */
@ReadAction
@Component("/feeds/offersLogsHistory")
@Category("feeds")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class GetFeedsOffersLogsHistoryAction extends AbstractUserVerifiedHostAction<Request, Response> {

    private final FeedsNative2YDao feedsNative2YDao;
    private final FeedsOffersLogsHistoryCHDao feedsOffersLogsHistoryCHDao;
    private final FeedsService feedsService;
    private final GoodsOffersLogsHistoryCHDao goodsOffersLogsHistoryCHDao;
    private final SerpdataLogsHistoryCHDao serpdataLogsHistoryCHDao;

    @Override
    public Response process(Request request) {
        String domain = WwwUtil.cutWWWAndM(request.getHostId());
        FeedsService.UrlWithUserInfo urlWithUserInfo = feedsService.parseUrl(request.url);
        String feedUrl = urlWithUserInfo.url();
        NativeFeedInfo2 info = feedsNative2YDao.get(domain, feedUrl);
        Preconditions.checkState(info != null);
        String feedDomain = WwwUtil.cutWWWAndM(IdUtils.urlToHostId(feedUrl));
        List<FeedRecord> offerRecords;
        final DateTime fromDate = ObjectUtils.max(request.fromDate, info.getAddDate());
        final DateTime toDate = ObjectUtils.max(request.toDate, info.getAddDate());
        if (info.getType() == NativeFeedType.STORES) {
            offerRecords = goodsOffersLogsHistoryCHDao.getHistory(domain,
                            Triple.of(info.getBusinessId(), info.getPartnerId(), info.getFeedId()), fromDate, toDate)
                    .stream().map(GoodsFeedRecord::toFeedRecord).collect(Collectors.toList());
        } else {
            offerRecords = feedsOffersLogsHistoryCHDao.getHistory(feedDomain, feedUrl, fromDate, toDate);
        }

        List<FeedRecord> serpData = serpdataLogsHistoryCHDao.getHistory(domain, feedUrl, fromDate, toDate);
        List<FeedHistoryRecord> result = new ArrayList<>();
        int serpIndex = 0;
        FeedRecord lastSerpData = serpData.isEmpty() ? null : serpData.get(0);
        for (FeedRecord record : offerRecords) {
            // search for first record of serpData with lte timestamp
            while (serpIndex < serpData.size() && serpData.get(serpIndex).getTimestamp().isAfter(record.getTimestamp())) {
                serpIndex++;
            }
            if (serpIndex < serpData.size()) {
                lastSerpData = serpData.get(serpIndex);
            }

            FeedStats offerStats = Optional.ofNullable(record.getStats()).orElse(FeedStats.EMPTY);
            FeedStats errorStats = Optional.ofNullable(record.getErrorStats()).orElse(FeedStats.EMPTY);
            FeedStats serpdataStats = Optional.ofNullable(lastSerpData).map(FeedRecord::getErrorStats).orElse(FeedStats.EMPTY);
            NativeFeedStatus status = feedsService.getStatus(info.getType(), record, lastSerpData);


            Map<FeedsValidationErrorEnum, List<OfferErrorInfo>> validationErrors = record.getErrors().entrySet().stream()
                    .map(pair -> Pair.of(FeedsValidationErrorEnum.byCode(pair.getKey()), pair.getValue()))
                    .filter(pair -> pair.getLeft() != FeedsValidationErrorEnum.Unknown)
                    .collect(W3Collectors.toEnumMap(FeedsValidationErrorEnum.class));
            Map<FeedsQualityErrorEnum, List<OfferErrorInfo>> qualityErrors = lastSerpData == null ? Collections.emptyMap() :
                    lastSerpData.getErrors().entrySet().stream()
                    .map(pair -> Pair.of(FeedsQualityErrorEnum.byCode(pair.getKey()), pair.getValue()))
                    .filter(pair -> pair.getLeft() != FeedsQualityErrorEnum.Unknown)
                    .collect(W3Collectors.toEnumMap(FeedsQualityErrorEnum.class));
            FeedHistoryRecord historyRecord = FeedHistoryRecord.builder()
                    .status(status)
                    .validationTimestamp(record.getTimestamp())
                    .qualityTimestamp(lastSerpData == null ? null : lastSerpData.getTimestamp())
                    .lastAccess(record.getLastAccess())
                    .totalOffers(offerStats.getTotal())
                    .validOffers(offerStats.getOk() + serpdataStats.getOk())
                    .errorOffers(offerStats.getError() + serpdataStats.getError())
                    .warningOffers(offerStats.getWarning() + serpdataStats.getWarning())
                    .validationErrorsTotal(Math.max(errorStats.getError() + errorStats.getWarning(), validationErrors.isEmpty() ? 0 : 1))
                    .qualityErrorsTotal(Math.max(serpdataStats.getError(), qualityErrors.isEmpty() ? 0 : 1))
                    .validationErrors(validationErrors)
                    .qualityErrors(qualityErrors).build();
            result.add(historyRecord);
        }
        result.sort(Comparator.comparing(FeedHistoryRecord::getLastAccess).reversed());

        DateTime sccTimestamp =
                info.getSccFinishTimestamp() != null ? info.getSccFinishTimestamp() :
                info.getSccTimestamp();

        return new Response(
                info.getStatusScc(),
                info.getStatusScc() == NativeFeedSccStatus.SUCCESS ?  Collections.emptyList() :
                        info.getErrorsScc().stream().filter(s -> !Strings.isNullOrEmpty(s)).collect(Collectors.toList()),
                sccTimestamp,
                result
        );
    }

    @Setter(onMethod_ = @RequestQueryProperty(required = true))
    public static final class Request extends AbstractUserVerifiedHostRequest {
        String url;
        DateTime fromDate;
        DateTime toDate;
    }

    @Value
    public static final class Response implements ActionResponse.NormalResponse {
        NativeFeedSccStatus statusScc;
        List<String> errorsScc;
        DateTime sccTimestamp;
        List<FeedHistoryRecord> records;
    }

    @Value
    @Builder
    public static final class FeedHistoryRecord {
        NativeFeedStatus status;
        DateTime validationTimestamp;
        DateTime qualityTimestamp;
        DateTime lastAccess;
        @Description("Всего офферов в прокачке")
        Long totalOffers;
        @Description("Валидных предложений в прокачке")
        Long validOffers;
        Long errorOffers;
        Long warningOffers;
        Long validationErrorsTotal;
        Long qualityErrorsTotal;
        @Description("Ошибки сгруппированные по коду")
        Map<FeedsValidationErrorEnum, List<OfferErrorInfo>> validationErrors;
        Map<FeedsQualityErrorEnum, List<OfferErrorInfo>> qualityErrors;
    }
}
