package ru.yandex.webmaster3.viewer.http.feeds.scc;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.feeds.mbi.MbiService;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

import static ru.yandex.webmaster3.core.feeds.feed.NativeFeedType.STORES;


/**
 * @author kravchenko99
 * @date 7/30/21
 */

@Component("/feeds/startScc")
@WriteAction
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Category("feeds")
public class RecheckSccAction extends Action<RecheckSccAction.Req, RecheckSccAction.Res> {
    private final FeedsNative2YDao feedsNative2YDao;
    private final MbiService mbiService;
    private final FeedsService feedsService;

    @Override
    public Res process(Req request) throws WebmasterException {
        WebmasterHostId hostId = request.getHostId();
        String domain = WwwUtil.cutWWWAndM(hostId);
        // temp parse url + login + password
        FeedsService.UrlWithUserInfo urlWithUserInfo = feedsService.parseUrl(request.url);
        request.url = urlWithUserInfo.url();
        //

        NativeFeedInfo2 nativeFeed2Info = feedsNative2YDao.get(domain, request.url);
        if (nativeFeed2Info == null) {
            return new Res.FeedNotExistsResponse();
        }
        if (request.type == STORES) {
            startMbiScc(nativeFeed2Info);
        } else {
            startUnisearchScc(nativeFeed2Info);
        }
        return new Res.NormalResponse();
    }

    public void startUnisearchScc(NativeFeedInfo2 info) {
        NativeFeedInfo2 newInfo = info
                .toBuilder()
                .sccTimestamp(DateTime.now())
                .statusScc(NativeFeedSccStatus.IN_PROGRESS)
                .build();
        feedsNative2YDao.update(newInfo);
    }

    public void startMbiScc(NativeFeedInfo2 info) {
        NativeFeedInfo2 newInfo = info
                .toBuilder()
                .statusScc(NativeFeedSccStatus.IN_PROGRESS)
                .build();
        mbiService.recheckShop(info.getPartnerId());
        feedsNative2YDao.update(newInfo);
    }

    @Setter(onMethod_ = @RequestQueryProperty)
    public static class Req extends AbstractUserVerifiedHostRequest {
        NativeFeedType type = STORES;
        String url;
        int regionId = 0;
    }

    public abstract static class Res implements ActionResponse {

        public static class NormalResponse extends Res implements ActionResponse.NormalResponse {
        }


        public enum ErrorCode {
            FEED_NOT_EXISTS,
        }

        public abstract static class AbstractStartSccError extends Res implements ActionResponse.ErrorResponse {
            private final ErrorCode code;

            public AbstractStartSccError(ErrorCode code) {
                this.code = code;
            }

            @Override
            public Enum<?> getCode() {
                return code;
            }
        }

        @Description("Такого фида с регионом нет")
        public static class FeedNotExistsResponse extends AbstractStartSccError {
            public FeedNotExistsResponse() {
                super(ErrorCode.FEED_NOT_EXISTS);
            }
        }
    }
}
