package ru.yandex.webmaster3.viewer.http.feeds.scc;

import java.util.Optional;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed_domain.FeedDomainInfo;
import ru.yandex.webmaster3.core.feeds.feed_domain.FeedDomainSccStatus;
import ru.yandex.webmaster3.core.feeds.mbi.AboService;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsDomainInfoYDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;


/**
 * @author kravchenko99
 * @date 7/27/22
 */

@Component("/feeds/fullScc/start")
@WriteAction
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Category("feeds")
public class StartFullSccAction extends Action<StartFullSccAction.Req, StartFullSccAction.Res> {

    private final FeedsDomainInfoYDao feedsDomainInfoYDao;
    private final AboService aboService;

    @Override
    public Res process(Req request) throws WebmasterException {
        WebmasterHostId hostId = request.getHostId();
        String domain = WwwUtil.cutWWWAndM(hostId);
        Optional<FeedDomainInfo> feedDomainInfo = feedsDomainInfoYDao.get(domain);
        if (feedDomainInfo.isEmpty()) {
            return new Res.BusinessForDomainNotExist();
        }
        Long businessId = feedDomainInfo.get().getBusinessId();
        aboService.startFullScc(businessId);
        FeedDomainInfo newDomainInfo = feedDomainInfo.get().toBuilder()
                .status(FeedDomainSccStatus.IN_PROGRESS)
                .startSccTimestamp(DateTime.now())
                .build();
        feedsDomainInfoYDao.update(newDomainInfo);

        return new Res.NormalResponse();
    }

    @Setter(onMethod_ = @RequestQueryProperty)
    public static class Req extends AbstractUserVerifiedHostRequest {
    }

    public abstract static class Res implements ActionResponse {

        public static class NormalResponse extends StartFullSccAction.Res implements ActionResponse.NormalResponse {
        }


        public enum ErrorCode {
            BUSINESS_NOT_EXISTS,
        }

        public abstract static class AbstractStartFullSccError extends StartFullSccAction.Res implements ActionResponse.ErrorResponse {
            private final ErrorCode code;

            public AbstractStartFullSccError(ErrorCode code) {
                this.code = code;
            }

            @Override
            public Enum<?> getCode() {
                return code;
            }
        }

        @Description("Такого фида с регионом нет")
        public static class BusinessForDomainNotExist extends StartFullSccAction.Res.AbstractStartFullSccError {
            public BusinessForDomainNotExist() {
                super(ErrorCode.BUSINESS_NOT_EXISTS);
            }
        }
    }
}
