package ru.yandex.webmaster3.viewer.http.feeds.statistics;

import java.util.EnumMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.feeds.feed.FeedsDefectRateErrorInfo;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsDefectRateErrorYDao;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsOfferBansYDao;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao.FeedRecord;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.SerpdataLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.models.DomainOffersStatistics;
import ru.yandex.webmaster3.storage.feeds.models.FeedStats;
import ru.yandex.webmaster3.storage.feeds.statistics.DomainOffersStatisticsYDao;
import ru.yandex.webmaster3.storage.feeds.statistics.FeedsDomainEnrichmentYDao;
import ru.yandex.webmaster3.storage.feeds.statistics.MarketOffersStatisticsYDao;
import ru.yandex.webmaster3.storage.feeds.statistics.MarketOffersStatisticsYDao.MarketOffersStatistics;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.feeds.statistics.GetFeedsDomainStatisticsAction.Request;
import ru.yandex.webmaster3.viewer.http.feeds.statistics.GetFeedsDomainStatisticsAction.Response;

/**
 * Created by Oleg Bazdyrev on 24/01/2022.
 */
@ReadAction
@Category("feeds")
@Component("/feeds/domainStatistics")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class GetFeedsDomainStatisticsAction extends AbstractUserVerifiedHostAction<Request, Response> {

    private final DomainOffersStatisticsYDao domainOffersStatisticsYDao;
    private final FeedsDefectRateErrorYDao feedsDefectRateErrorYDao;
    private final FeedsDomainEnrichmentYDao feedsDomainEnrichmentYDao;
    private final FeedsNative2YDao feedsNative2YDao;
    private final FeedsOfferBansYDao feedsOfferBansYDao;
    private final FeedsOffersLogsHistoryCHDao feedsOffersLogsHistoryCHDao;
    private final GoodsOffersLogsHistoryCHDao goodsOffersLogsHistoryCHDao;
    private final MarketOffersStatisticsYDao marketOffersStatisticsYDao;
    private final SerpdataLogsHistoryCHDao serpdataLogsHistoryCHDao;
    private final SettingsService settingsService;

    @Override
    public Response process(Request request) {
        // собираем статистику и ошибки
        String domain = WwwUtil.cutWWWAndM(request.getHostId());
        Map<String, NativeFeedInfo2> feeds = feedsNative2YDao.list(domain).stream().collect(
                Collectors.toMap(NativeFeedInfo2::getUrl, Function.identity(), W3Collectors.replacingMerger()));
        LocalDate lastUpdate = LocalDate.parse(settingsService.getSettingCached(CommonDataType.LAST_FEEDS_OFFERS_STATISTICS_IMPORT).getValue());

        Map<NativeFeedType, DomainOffersStatistics> stats = domainOffersStatisticsYDao.getStatistics(lastUpdate, domain)
                .stream().collect(Collectors.toMap(DomainOffersStatistics::getType, Function.identity()));
        Map<Pair<String, NativeFeedType>, FeedRecord> serpdataStats = serpdataLogsHistoryCHDao.getLastState(domain, null, null)
                .stream().collect(Collectors.toMap(
                        fr -> Pair.of(fr.getUrl(), NativeFeedType.fromCode(fr.getType())), Function.identity(), W3Collectors.replacingMerger()));
        Map<Pair<String, NativeFeedType>, FeedRecord> offerbaseStats = feedsOffersLogsHistoryCHDao.getLastState(feeds.keySet())
                .stream().collect(Collectors.toMap(
                        fr -> Pair.of(fr.getUrl(), feeds.get(fr.getUrl()).getType()), Function.identity(), W3Collectors.replacingMerger()));
        var feedIds = feeds.values().stream().filter(f -> f.getType() == NativeFeedType.STORES)
                .filter(f -> f.getBusinessId() != null && f.getPartnerId() != null && f.getFeedId() != null)
                .map(f -> Triple.of(f.getBusinessId(), f.getPartnerId(), f.getFeedId()))
                .collect(Collectors.toList());
        goodsOffersLogsHistoryCHDao.getLastState(feedIds).forEach(goodsFeedRecord -> {
            offerbaseStats.put(Pair.of(goodsFeedRecord.getFeedUrl(), NativeFeedType.STORES), goodsFeedRecord.toFeedRecord());
        });

        Map<NativeFeedType, FeedsDefectRateErrorInfo> drStatuses = feedsDefectRateErrorYDao.list(domain).stream()
            .collect(Collectors.toMap(FeedsDefectRateErrorInfo::getType ,Function.identity()));

        EnumMap<NativeFeedType, DomainStatistics> statisticsByType = new EnumMap<>(NativeFeedType.class);
        Map<NativeFeedType, Long> bansCount = feedsOfferBansYDao.countByType(domain);
        // enrichment
        LocalDate lsp = LocalDate.parse(settingsService.getSettingCached(CommonDataType.LAST_PROCESSED_USER_SESSIONS_FEEDS).getValue());
        Map<NativeFeedType, Double> domainEnrichment = feedsDomainEnrichmentYDao.getEnrichmentInfo(domain, lsp.minusDays(7), lsp);
        Map<NativeFeedType, Double> totalEnrichment = feedsDomainEnrichmentYDao.getEnrichmentInfo("", lsp.minusDays(7), lsp);

        feeds.values().stream().map(NativeFeedInfo2::getType).forEach(
                key -> statisticsByType.put(key,
                        new DomainStatistics(key, stats.get(key), bansCount.getOrDefault(key, 0L), domainEnrichment.get(key), totalEnrichment.get(key)))
        );
        for (NativeFeedInfo2 feedInfo : feeds.values()) {
            // TODO temp
            if (feedInfo.getType() == NativeFeedType.STORES) {
                continue;
            }
            FeedsDefectRateErrorInfo drStatus = drStatuses.get(feedInfo.getType());
            // TODO computeIfAbsent -> get
            DomainStatistics domainStatistics = statisticsByType.get(feedInfo.getType());
            Pair<String, NativeFeedType> feedAndType = Pair.of(feedInfo.getUrl(), feedInfo.getType());
            FeedStats serpStats = Optional.ofNullable(serpdataStats.get(feedAndType)).map(FeedRecord::getStats).orElse(FeedStats.EMPTY);
            FeedStats offStats = Optional.ofNullable(offerbaseStats.get(feedAndType)).map(FeedRecord::getStats).orElse(FeedStats.EMPTY);
            domainStatistics.offers.total += offStats.getError();
            domainStatistics.offers.errors += offStats.getError() + serpStats.getError();
        }
        // domain serpstat errors
        for (var entry : statisticsByType.entrySet()) {
            FeedStats domainSerpStats = Optional.ofNullable(serpdataStats.get(Pair.of("", entry.getKey()))).map(FeedRecord::getStats).orElse(FeedStats.EMPTY);
            DomainStatistics value = entry.getValue();
            if (value == null || value.sets == null) {
                continue;
            }
            //value.sets.total += domainSerpStats.getError() + domainSerpStats.getWarning();
            value.sets.errors += domainSerpStats.getError() + domainSerpStats.getWarning();
            value.sets.errors = Math.min(value.sets.errors, value.sets.total);
            value.sets.valid = value.sets.total - value.sets.errors;
        }
        LocalDate lastMarketUpdate = LocalDate.parse(settingsService.getSettingCached(CommonDataType.LAST_MARKET_OFFERS_STATISTICS_IMPORT).getValue());
        // collect goods feeds
        var ids = feeds.values().stream().filter(f -> f.getType() == NativeFeedType.STORES)
                .collect(Collectors.toMap(f -> Pair.of(f.getBusinessId(), f.getPartnerId()), Function.identity()));
        for (MarketOffersStatistics mos : marketOffersStatisticsYDao.getStatistics(lastMarketUpdate, ids.keySet())) {
            // TODO computeIfAbsent -> get
            DomainStatistics domainStatistics = statisticsByType.get(NativeFeedType.STORES);
            NativeFeedInfo2 feedInfo = ids.get(Pair.of(mos.getBusinessId(), mos.getPartnerId()));
            FeedStats offStats = Optional.ofNullable(offerbaseStats.get(Pair.of(feedInfo.getUrl(), feedInfo.getType())))
                    .map(FeedRecord::getStats).orElse(FeedStats.EMPTY);
            domainStatistics.offers.total += mos.getOffers() + offStats.getError() + offStats.getWarning();
            domainStatistics.offers.auto += mos.getRobotOffers();
            domainStatistics.offers.errors += offStats.getError() + offStats.getWarning();
            domainStatistics.offers.valid += mos.getOffers() + mos.getRobotOffers();
        }

        return new Response(lastUpdate, statisticsByType);
    }

    public static final class Request extends AbstractUserVerifiedHostRequest {

    }

    @Value
    public static class Response implements ActionResponse.NormalResponse {
        LocalDate lastUpdate;
        EnumMap<NativeFeedType, DomainStatistics> statisticsByType;
    }

    @Value
    public static class DomainStatistics {
        Statistics offers;
        Statistics sets;
        long offerBansCount;
        Double enrichment;
        Double totalEnrichment;

        public DomainStatistics(NativeFeedType type, DomainOffersStatistics dos, long offerBansCount, Double enrichment, Double totalEnrichment) {
            if (type == NativeFeedType.STORES) {
                offers = new Statistics(0L, 0L, 0L, 0L);
                sets = null;
            } else if (dos != null) {
                offers = new Statistics(dos.getOffers(), dos.getOffers(), null, 0L);
                sets = new Statistics(dos.getSets(), dos.getSets(), null, 0L);
            } else {
                offers = new Statistics(0L, 0L, null, 0L);
                sets = new Statistics(0L, 0L, null, 0L);
            }
            this.offerBansCount = offerBansCount;
            this.enrichment = enrichment;
            this.totalEnrichment = totalEnrichment;
        }
    }

    @Getter
    @AllArgsConstructor
    public static class Statistics {
        @Description("Всего")
        Long total;
        @Description("Корректных")
        Long valid;
        @Description("От робота (только офферы)")
        Long auto;
        @Description("Ошибок")
        Long errors;
    }


}
