package ru.yandex.webmaster3.viewer.http.host;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.events2.HostEventId;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.postpone.PostWelcomeEmailData;
import ru.yandex.webmaster3.storage.postpone.PostponeActionYDao;
import ru.yandex.webmaster3.storage.postpone.PostponeOperationType;
import ru.yandex.webmaster3.storage.user.dao.UserHostVerificationYDao;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.storage.verification.HostVerifierService;
import ru.yandex.webmaster3.viewer.http.AbstractUserAction;

/**
 * @author avhaliullin
 */
@Description("Добавить хост")
@WriteAction
@Category("verification")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Slf4j
@Component("/user/host/add")
public class AddHostAction extends AbstractUserAction<AddHostRequest, AddHostResponse> {

    private static final int HOST_LIST_LIMIT = 1703;

    private final MirrorService2 mirrorService2;
    private final HostVerifierService hostVerifierService;
    private final UserHostVerificationYDao userHostVerificationYDao;
    private final UserHostsService userHostsService;
    private final PostponeActionYDao postponeActionYDao;

    @Override
    public AddHostResponse process(AddHostRequest request) {
        WebmasterHostId hostId;
        WebmasterUser user = request.getWebmasterUser();
        try {
            hostId = IdUtils.urlToHostIdWithUpgradedValidation(request.getHostName());
            //TODO: Это костыль. SupportedProtocols.getURL() недостаточно валидирует имя хоста. В будущем нужно
            // отказаться от SupportedProtocols в пользу чего-то рабочего
            if (!hostId.getPunycodeHostname().contains(".")) {
                throw new RuntimeException("Invalid host name " + hostId.getPunycodeHostname());
            }
        } catch (RuntimeException e) {
            throw new WebmasterException("Passed bad host name",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "hostName",
                            request.getHostName()), e);
        }
        try {
            UserHostVerificationInfo verificationInfo;
            // Проверяем, не добавлен ли хост уже
            if (userHostsService.getVerifiedHost(user, hostId) != null) {
                return new AddHostResponse.HostAlreadyAddedError(hostId, true);
            }
            verificationInfo = userHostVerificationYDao.getLatestRecord(request.getUserId(), hostId);
            if (verificationInfo != null && verificationInfo.isAddedToList()) {
                return new AddHostResponse.HostAlreadyAddedError(hostId, false);
            }

            if (!request.isForceNotMainMirror()) {
                WebmasterHostId mainMirrorId = mirrorService2.getMainMirrorFromAllMirrors(hostId);

                if (!mainMirrorId.equals(hostId)) {
                    boolean mainMirrorAdded;
                    boolean mainMirrorVerified;
                    if (userHostsService.getVerifiedHost(user, mainMirrorId) != null) {
                        mainMirrorAdded = true;
                        mainMirrorVerified = true;
                    } else {
                        mainMirrorVerified = false;
                        mainMirrorAdded = userHostsService.getUnverifiedHost(user, mainMirrorId) != null;
                    }
                    return new AddHostResponse.NotMainMirrorError(hostId, mainMirrorId, mainMirrorAdded,
                            mainMirrorVerified);
                }
            }

            if (userHostsService.getAllHostsCount(user) >= HOST_LIST_LIMIT) {
                return new AddHostResponse.HostsLimitExceededError(HOST_LIST_LIMIT);
            }

            HostEventId eventId = HostEventId.createFromUserRequest(request.getBalancerRequestId(),
                    request.getUserId());
            boolean autoVerified = hostVerifierService.addHostUinInitRecord(user, hostId, verificationInfo, eventId);
            var postWelcomeEmailData = new PostWelcomeEmailData(user.getUserId(), hostId, LanguageEnum.RU);
            postponeActionYDao.insert(DateTime.now().withTimeAtStartOfDay().toDateTime(DateTimeZone.UTC),
                    PostponeOperationType.POST_WELCOME_EMAIL, JsonMapping.writeValueAsString(postWelcomeEmailData));

            return new AddHostResponse.NormalResponse(hostId, autoVerified);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to add host",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }
}
