package ru.yandex.webmaster3.viewer.http.host;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestTimeout;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructure;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructureNode;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.checklist.data.SummarySiteProblemsInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.service.DisplayNameService2;
import ru.yandex.webmaster3.storage.host.service.CommercialOwnersService;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.sitestructure.SiteStructureService;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserAddedHostAction;
import ru.yandex.webmaster3.viewer.http.common.response.CommonHostInfo;
import ru.yandex.webmaster3.viewer.http.common.response.HostDataStatus;
import ru.yandex.webmaster3.viewer.http.user.GetHostListAction;

/**
 * @author aherman
 */
@ReadAction
@Category("verification")
@RequestTimeout(15)
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/host/info")
public class GetHostInfoAction extends AbstractUserAddedHostAction<GetHostInfoRequest, GetHostInfoResponse> {

    private final CommercialOwnersService commercialOwnersService;
    private final DisplayNameService2 displayNameService2;
    private final MirrorService2 mirrorService2;
    private final SiteProblemsService siteProblemsService;
    private final SiteStructureService siteStructureService;
    private final UserHostsService userHostsService;

    @Override
    public GetHostInfoResponse process(GetHostInfoRequest request) {
        boolean hostPinned = userHostsService.isHostPinned(request.getWebmasterUser(), request.getHostId());

        HostDataStatus hostDataStatus = getHostStatus(request.getWebmasterUser(), request.getHostId());
        SummarySiteProblemsInfo rtProblems = siteProblemsService.getRealTimeProblemsSummary(request.getHostId());
        String displayName = displayNameService2.getHostUrlWithDisplayName(request.getHostId());

        CommonHostInfo commonHostInfo = GetHostListAction.toCommonHostInfo(request.getHostId(), displayName, rtProblems, hostPinned);

        WebmasterHostId mainMirrorHostId = mirrorService2.getMainMirror(request.getHostId());

        if (mainMirrorHostId == null){
            mainMirrorHostId = request.getHostId();
        }
        HostDataStatus mainMirrorStatus = getHostStatus(request.getWebmasterUser(), mainMirrorHostId);

        boolean isCommercial = commercialOwnersService.isCommercialOwner(IdUtils.urlToHostId(WwwUtil.cutWWWAndM(request.getHostId())));

        return new GetHostInfoResponse.NormalResponse(
                commonHostInfo,
                hostDataStatus,
                mainMirrorHostId,
                mainMirrorStatus != HostDataStatus.HOST_NOT_ADDED && mainMirrorStatus != HostDataStatus.HOST_NOT_VERIFIED,
                isCommercial
        );
    }

    private HostDataStatus getHostStatus(WebmasterUser user, WebmasterHostId hostId){
        UserVerifiedHost verifiedHost = userHostsService.getVerifiedHost(user, hostId);
        HostDataStatus hostDataStatus = null;

        if (verifiedHost == null) {
            if (userHostsService.getUnverifiedHost(user, hostId) == null) {
                hostDataStatus = HostDataStatus.HOST_NOT_ADDED;
            } else {
                hostDataStatus = HostDataStatus.HOST_NOT_VERIFIED;
            }
        } else {
            NewSiteStructureNode node = siteStructureService.getNode(hostId, NewSiteStructure.ROOT_NODE_ID);
            if (node == null) {
                hostDataStatus = HostDataStatus.HOST_NOT_LOADED;
            }
        }

        return hostDataStatus;
    }
}
