package ru.yandex.webmaster3.viewer.http.host;

import java.util.Collections;
import java.util.List;

import lombok.AllArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.iks.data.Sqi;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.metrika.counters.MetrikaCountersUtil;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.checklist.data.SummarySiteProblemsInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.host.HostDataState;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.service.DisplayNameService2;
import ru.yandex.webmaster3.storage.iks.IksService;
import ru.yandex.webmaster3.storage.metrika.data.MetrikaDomainCrawlState;
import ru.yandex.webmaster3.storage.sitestructure.SiteStructureService;
import ru.yandex.webmaster3.storage.turbo.service.TurboSearchUrlsStatisticsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.user.GetHostListAction;
import ru.yandex.webmaster3.viewer.http.user.GetHostListResponse;

/**
 * @author aherman
 */

@ReadAction
@Category("verification")
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class GetHostStatisticsAction extends AbstractUserVerifiedHostAction<GetHostStatisticsRequest, GetHostStatisticsResponse> {

    private final AbtService abtService;
    private final DisplayNameService2 displayNameService2;
    private final IksService iksService;
    private final SiteProblemsService siteProblemsService;
    private final SiteStructureService siteStructureService;
    private final TurboSearchUrlsStatisticsService turboSearchUrlsStatisticsService;
    private final FeedsService feedsService;

    @Override
    public GetHostStatisticsResponse process(GetHostStatisticsRequest request) {

        WebmasterHostId hostId = request.getHostId();
        HostDataState currentHostDataState = siteStructureService.getHostDataState(hostId);
        SummarySiteProblemsInfo rpSummary = siteProblemsService.getRealTimeProblemsSummary(hostId);
        Pair<Boolean, String> ownerInfo = IksService.comparePunycodeHostNameWithPunycodeOwnerName(hostId,
                iksService.getMascotOwner(hostId));

        Sqi sqi = iksService.getSqi(hostId).orElse(IksService.defaultSqi(ownerInfo.getRight()));
        String domain = WwwUtil.cutWWWAndM(hostId);
        String metrikaDomain = MetrikaCountersUtil.hostToPunycodeDomain(hostId);

        final TurboSearchUrlsStatisticsService.TurboDomainStats turboInfo = turboSearchUrlsStatisticsService.getTurboInfo(domain);

        String displayName = displayNameService2.getHostUrlWithDisplayName(hostId);
        boolean hasSuccessfulGoodFeeds = feedsService.getDomainsWithTvInSearch(List.of(domain)).size() > 0;

        GetHostListResponse.ShortHostInfo shortHostInfo = GetHostListAction.toResponseInfo(
                hostId, false, true, true, false,
                currentHostDataState, rpSummary,
                Collections.emptyList(), ownerInfo, sqi,
                turboInfo, new MetrikaDomainCrawlState(metrikaDomain), displayName, hasSuccessfulGoodFeeds);

        return new GetHostStatisticsResponse.NormalResponse(shortHostInfo);
    }
}
