package ru.yandex.webmaster3.viewer.http.host.delegation;

import java.util.Arrays;

import com.datastax.driver.core.utils.UUIDs;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.blackbox.UserWithLogin;
import ru.yandex.webmaster3.core.blackbox.service.BlackboxUsersService;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationType;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.delegation.HostDelegationRecord;
import ru.yandex.webmaster3.storage.delegation.UserDailyDelegationsYDao;
import ru.yandex.webmaster3.storage.delegation.UserDelegationsForSendYDao;
import ru.yandex.webmaster3.storage.delegation.UserHostDelegationsYDao;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.postpone.PostWelcomeEmailData;
import ru.yandex.webmaster3.storage.postpone.PostponeActionYDao;
import ru.yandex.webmaster3.storage.postpone.PostponeOperationType;
import ru.yandex.webmaster3.storage.user.dao.UserHostVerificationYDao;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author avhaliullin
 */
@Description("Делегировать права на сайт другому пользователю")
@WriteAction
@Category("verification")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class DelegateHostAction extends AbstractUserVerifiedHostAction<DelegateHostRequest, DelegateHostResponse> {
    private static final int DELEGATIONS_DAILY_LIMIT = 150;

    private final UserHostDelegationsYDao userHostDelegationsYDao;
    private final UserDailyDelegationsYDao userDailyDelegationsYDao;
    private final UserDelegationsForSendYDao userDelegationsForSendYDao;
    private final BlackboxUsersService blackboxExternalYandexUsersService;
    private final UserHostVerificationYDao userHostVerificationYDao;
    private final UserHostsService userHostsService;
    private final WMCEventsService wmcEventsService;
    private final PostponeActionYDao postponeActionYDao;

    @Override
    public DelegateHostResponse process(DelegateHostRequest request) {
        if (!request.getUserVerifiedHost().getVerificationType().isCanBeDelegated()) {
            return new DelegateHostResponse.NotDelegatableError();
        }
        if (userDailyDelegationsYDao.countDelegations(request.getUserId()) >= DELEGATIONS_DAILY_LIMIT) {
            return new DelegateHostResponse.DailyDelegationsLimitExceededError(DELEGATIONS_DAILY_LIMIT);
        }
        UserWithLogin toUser = blackboxExternalYandexUsersService.getUserByLogin(request.getToUserLogin());
        if (toUser == null) {
            return new DelegateHostResponse.UserNotFoundError();
        }

        boolean cheatVerifiedByUser = false;
        {
            UserVerifiedHost userVerifiedHost = userHostsService.getVerifiedHost(new WebmasterUser(toUser.getUserId()), request.getHostId());
            // если права у пользователя делегированы то даем переделегировать через другого пользователя
            if (userVerifiedHost != null && userVerifiedHost.getVerificationType() != VerificationType.DELEGATED) {
                if (userVerifiedHost.getVerificationType().isCanBeDelegated()) {
                    return new DelegateHostResponse.AlreadyVerifiedError();
                } else {
                    cheatVerifiedByUser = true;
                }
            }
        }
        {
            HostDelegationRecord delegationRecord =
                    userHostDelegationsYDao.getDelegationForHostToUser(request.getHostId(), toUser.getUserId());
            if (delegationRecord != null && !delegationRecord.isCancelled() && delegationRecord.getFromUser() == request.getUserId()) {
                return new DelegateHostResponse.AlreadyDelegatedError();
            }
        }

        UserHostVerificationInfo verificationInfo =
                userHostVerificationYDao.getLatestRecord(toUser.getUserId(), request.getHostId());
        userDailyDelegationsYDao.addDelegation(request.getUserId());
        userHostDelegationsYDao.addDelegationRecord(
                new HostDelegationRecord(
                        request.getUserId(),
                        toUser.getUserId(),
                        request.getHostId(),
                        DateTime.now(),
                        false, null, null)
        );

        boolean hostAddedByUser = false;
        if (!cheatVerifiedByUser && verificationInfo != null && verificationInfo.isAddedToList()) {
            hostAddedByUser = true;
            verificationInfo = verificationInfo.copyDelegated(UUIDs.timeBased());
            userHostsService.addVerifiedHost(
                    new WebmasterUser(toUser.getUserId()),
                    new UserVerifiedHost(
                            request.getHostId(),
                            DateTime.now(),
                            verificationInfo.getLastAttempt().plus(VerificationType.DELEGATED.getReverifyPeriod()).toDateTime(),
                            verificationInfo.getVerificationUin(),
                            VerificationType.DELEGATED
                    )
            );
            userHostVerificationYDao.addVerificationRecord(verificationInfo);
        }

        wmcEventsService.addEvent(
                new RetranslateToUsersEvent<>(
                        new UserHostMessageEvent<>(
                                request.getHostId(),
                                toUser.getUserId(),
                                new MessageContent.HostAccessDelegatedSomeoneElse(
                                        request.getHostId()),
                                NotificationType.SITE_ACCESS,
                                false),
                        Arrays.asList(request.getUserId(), toUser.getUserId())
                )
        );


        userDelegationsForSendYDao.add(toUser.getUserId(), request.getHostId());
        var postWelcomeEmailData = new PostWelcomeEmailData(toUser.getUserId(), request.getHostId(), LanguageEnum.RU);
        postponeActionYDao.insert(DateTime.now().withTimeAtStartOfDay().toDateTime(DateTimeZone.UTC), PostponeOperationType.POST_WELCOME_EMAIL, JsonMapping.writeValueAsString(postWelcomeEmailData));

        return new DelegateHostResponse.NormalResponse(toUser.getUserId(), hostAddedByUser);
    }

}
