package ru.yandex.webmaster3.viewer.http.host.verification;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationFailEnum;
import ru.yandex.webmaster3.core.host.verification.VerificationType;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.delegation.HostDelegationRecord;
import ru.yandex.webmaster3.storage.delegation.UserHostDelegationsYDao;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.storage.verification.HostVerifierService;
import ru.yandex.webmaster3.viewer.http.AbstractUserAddedHostAction;

/**
 * @author avhaliullin
 */
@Description("Посмотреть информацию о подтверждении прав")
@ReadAction
@Category("verification")
public class HostVerificationInfoAction extends AbstractUserAddedHostAction<HostVerificationInfoRequest, HostVerificationInfoResponse> {
    private static final Logger log = LoggerFactory.getLogger(HostVerificationInfoAction.class);

    private UserHostsService userHostsService;
    private UserHostDelegationsYDao userHostDelegationsYDao;

    @Override
    public HostVerificationInfoResponse process(HostVerificationInfoRequest request) {
        if (request.getUserAddedHost().isVerified()) {
            UserVerifiedHost verifiedHost = userHostsService.getVerifiedHost(request.getWebmasterUser(), request.getHostId());
            if (verifiedHost.getVerificationType() == VerificationType.DELEGATED) {
                try {
                    HostDelegationRecord delegationRecord = userHostDelegationsYDao.getDelegationForHostToUser(request.getHostId(), request.getUserId());
                    Long fromUserId = null;
                    if (delegationRecord == null) {
                        log.error("No delegation record found for user " + request.getUserId() + " host " + request.getHostId());
                    } else {
                        fromUserId = delegationRecord.getFromUser();
                    }
                    return HostVerificationInfoResponse.createDelegated(verifiedHost, fromUserId);
                } catch (WebmasterYdbException e) {
                    throw new WebmasterException("Failed to get user delegation info",
                            new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
                }
            }
            return HostVerificationInfoResponse.createVerified(verifiedHost);
        } else {
            UserHostVerificationInfo verificationInfo =
                    userHostsService.getVerificationInfo(request.getUserId(), request.getHostId());
            if (verificationInfo.getVerificationStatus() == null) {
                return HostVerificationInfoResponse.createNeverVerified(verificationInfo);
            } else {
                if (HostVerifierService.isVerificationExpired(verificationInfo)) {
                    return HostVerificationInfoResponse.createExpired(verificationInfo);
                }
                //TODO: Workaround WMC-2873
                try {
                    if (verificationInfo.getVerificationFailInfo() != null &&
                            verificationInfo.getVerificationFailInfo().getType() == VerificationFailEnum.SELF_VERIFICATION_CANCELLED) {
                        List<UserHostVerificationInfo> records =
                                userHostsService.getAllVerificationRecords(request.getUserId(), request.getHostId());
                        for (UserHostVerificationInfo record : records) {
                            if (record.getVerificationFailInfo() != null &&
                                    record.getVerificationFailInfo().getType() == VerificationFailEnum.TXT_VERIFICATION_NOT_SUPPORTED) {
                                return HostVerificationInfoResponse.createNotVerified(
                                        verificationInfo
                                                .copyAsVerificationStartRecord(
                                                        verificationInfo.getRecordId(),
                                                        VerificationType.HTML_FILE,
                                                        record.getVerificationCausedBy()
                                                )
                                                .copyWithTotallyFail(
                                                        verificationInfo.getRecordId(),
                                                        record.getVerificationFailInfo()
                                                )
                                );
                            }
                        }
                    }
                } catch (Exception e) {
                    log.error("Failed to check if it's cancelled txt verification", e);
                }
                return HostVerificationInfoResponse.createNotVerified(verificationInfo);
            }
        }
    }

    @Required
    public void setUserHostsService(UserHostsService userHostsService) {
        this.userHostsService = userHostsService;
    }

    @Required
    public void setUserHostDelegationsYDao(UserHostDelegationsYDao userHostDelegationsYDao) {
        this.userHostDelegationsYDao = userHostDelegationsYDao;
    }
}
