package ru.yandex.webmaster3.viewer.http.host.verification;

import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.core.worker.task.VerifyHostTaskData;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskPriority;
import ru.yandex.webmaster3.storage.delegation.HostDelegationRecord;
import ru.yandex.webmaster3.storage.delegation.UserHostDelegationsYDao;
import ru.yandex.webmaster3.storage.user.dao.UserHostVerificationYDao;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author avhaliullin
 */
@Description("Сбросить права пользователю")
@WriteAction
@Category("verification")
@Component("/user/host/verification/reset")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ResetVerificationAction extends AbstractUserVerifiedHostAction<ResetVerificationRequest, ResetVerificationResponse> {

    private final UserHostsService userHostsService;
    private final UserHostVerificationYDao userHostVerificationYDao;
    private final WorkerClient workerClient;
    private final UserHostDelegationsYDao userHostDelegationsYDao;

    @Override
    public ResetVerificationResponse process(ResetVerificationRequest request) {
        if (request.getUserVerifiedHost().getVerificationType().isCanBeDelegated()) {
            HostDelegationRecord delegationRecord =
                    userHostDelegationsYDao.getDelegationForHostToUser(request.getHostId(), request.getForUserId());
            if (delegationRecord != null && !delegationRecord.isCancelled()) {
                userHostDelegationsYDao.addDelegationRecord(delegationRecord.copyCancelled(request.getUserId()));
            }
        }
        UserVerifiedHost userVerifiedHost = userHostsService.getVerifiedHost(
                new WebmasterUser(request.getForUserId()),
                request.getHostId()
        );
        if (userVerifiedHost == null) {
            return new ResetVerificationResponse();
        }
        UUID recordId = UUIDs.timeBased();
        userHostVerificationYDao.addVerificationRecord(UserHostVerificationInfo.createVerificationStartRecord(
                recordId, request.getForUserId(), request.getHostId(), userVerifiedHost.getVerificationType(),
                userVerifiedHost.getVerificationUin(), VerificationCausedBy.RESET_REQUESTED, request.getUserId()
        ));
        workerClient.enqueueTask(
                VerifyHostTaskData.createTaskData(
                        request.getHostId(),
                        request.getForUserId(),
                        recordId,
                        userVerifiedHost.getVerificationType(),
                        WorkerTaskPriority.HIGH
                )
        );
        return new ResetVerificationResponse();
    }
}
