package ru.yandex.webmaster3.viewer.http.host.verification;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.host.verification.VerificationType;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.storage.delegation.HostDelegationRecord;
import ru.yandex.webmaster3.storage.delegation.UserHostDelegationsYDao;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author avhaliullin
 */
@Description("Посмотреть всех пользователей, подтвердивших хост")
@ReadAction
@Category("verification")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UsersVerifiedHostAction extends AbstractUserVerifiedHostAction<UsersVerifiedHostRequest, UsersVerifiedHostResponse> {
    private static final Logger log = LoggerFactory.getLogger(UsersVerifiedHostAction.class);

    private final UserHostDelegationsYDao userHostDelegationsYDao;
    private final UserHostsService userHostsService;

    @Override
    public UsersVerifiedHostResponse process(UsersVerifiedHostRequest request) {
        Map<Long, UserVerifiedHost> usersVerifiedHost =
                userHostsService.listUsersVerifiedHost(request.getHostId());
        Map<Long, UsersVerifiedHostResponse.UserInfo> result = new HashMap<>();
        for (Map.Entry<Long, UserVerifiedHost> entry : usersVerifiedHost.entrySet()) {
            long userId = entry.getKey();
            UserVerifiedHost userVerifiedHost = entry.getValue();
            if (!userVerifiedHost.getVerificationType().isDisplayable()) {
                continue;
            }
            Long delegatedByUserId = null;
            if (userVerifiedHost.getVerificationType() == VerificationType.DELEGATED) {
                HostDelegationRecord delegationRecord =
                        userHostDelegationsYDao.getDelegationForHostToUser(request.getHostId(), userId);
                if (delegationRecord == null) {
                    log.error("No delegation record found for user " + userId + " host " + request.getHostId());
                } else {
                    delegatedByUserId = delegationRecord.getFromUser();
                }
            }
            result.put(userId, new UsersVerifiedHostResponse.UserInfo(userId, userVerifiedHost.getVerificationType(),
                    userVerifiedHost.getVerificationDate(), userVerifiedHost.getVerificationUin(), delegatedByUserId));
        }
        List<HostDelegationRecord> delegationRecords = userHostDelegationsYDao.getDelegationsForHost(request.getHostId());
        for (HostDelegationRecord delegationRecord : delegationRecords) {
            if (!delegationRecord.isCancelled() && !result.containsKey(delegationRecord.getToUser())
                    && result.containsKey(delegationRecord.getFromUser())) {
                result.put(delegationRecord.getToUser(),
                        new UsersVerifiedHostResponse.UserInfo(delegationRecord.getToUser(), VerificationType.DELEGATED,
                                delegationRecord.getDelegatedDate(), 0L, delegationRecord.getFromUser()));
            }
        }
        return new UsersVerifiedHostResponse(
                result.values()
                        .stream()
                        .sorted(Comparator.comparing(UsersVerifiedHostResponse.UserInfo::getVerificationDate))
                        .collect(Collectors.toList())
        );
    }

}
