package ru.yandex.webmaster3.viewer.http.host.verification;

import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.host.verification.VerificationStatus;
import ru.yandex.webmaster3.core.host.verification.VerificationTypeScope;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.user.dao.UserHostVerificationYDao;
import ru.yandex.webmaster3.storage.verification.HostVerifierService;
import ru.yandex.webmaster3.viewer.http.AbstractUserAddedHostAction;

/**
 * @author avhaliullin
 */
@Description("Запустить процедуру подтверждения прав на хост")
@WriteAction
@Category("verification")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class VerifyHostAction extends AbstractUserAddedHostAction<VerifyHostRequest, VerifyHostResponse> {
    private static final Logger log = LoggerFactory.getLogger(VerifyHostAction.class);

    private final UserHostVerificationYDao userHostVerificationYDao;
    private final HostVerifierService hostVerifierService;

    @Override
    public VerifyHostResponse process(VerifyHostRequest request) {
        if (!request.getVerificationType().visibleIn(VerificationTypeScope.VIEWER)) {
            return new VerifyHostResponse.VerificationNotApplicable();
        }

        if (request.getUserAddedHost().isVerified()) {
            return new VerifyHostResponse.AlreadyVerifiedError();
        }

        if (!hostVerifierService.isApplicable(request.getHostId(), request.getUserId(), request.getVerificationType())) {
            return new VerifyHostResponse.VerificationNotApplicable();
        }

        UserHostVerificationInfo verificationInfo = userHostVerificationYDao.getLatestRecord(request.getUserId(), request.getHostId());

        if (verificationInfo.getVerificationStatus() != null) {
            if (HostVerifierService.isVerificationInProgress(verificationInfo)) {
                return new VerifyHostResponse.VerificationIsInProgressError();
            }

            if (verificationInfo.getVerificationStatus() == VerificationStatus.VERIFIED) {
                log.error("Host " + request.getHostId() + " unverified for user " + request.getUserId() +
                        " has verification record " + verificationInfo.getRecordId());
            }
        }

        boolean launched = hostVerifierService.initiateVerification(
                verificationInfo,
                VerificationCausedBy.INITIAL_VERIFICATION,
                request.getHostId(),
                request.getUserId(),
                request.getVerificationType()
        );
        return launched
                ? new VerifyHostResponse.NormalResponse()
                : new VerifyHostResponse.VerificationNotApplicable();
    }
}
