package ru.yandex.webmaster3.viewer.http.importanturls;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.importanturls.ImportantUrlsService;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrl;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrlRequest;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.importanturls.AddImportantUrlsResponse.WarningType;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author Oleg Bazdyrev
 * WMC-3620
 */
@WriteAction
@Description("Добавление одной или нескольких подписок на страницы в пределах одного хоста")
@Category("importanturls")
public class AddImportantUrlsAction extends
        AbstractUserVerifiedHostAction<AddImportantUrlsRequest, AddImportantUrlsResponse> {

    private ImportantUrlsService importantUrlsService;

    @Override
    public AddImportantUrlsResponse process(AddImportantUrlsRequest request) {
        // преобразуем URL к стандартному виду
        Set<String> importantRelUrls = new LinkedHashSet<>();
        List<String> invalidUrls = new ArrayList<>();
        WebmasterHostId hostId = request.getHostId();
        for (String urlString : request.getUrls()) {
            try {
                if (StringUtils.isBlank(urlString)) {
                    continue;
                }
                String relativeUrl = IdUtils.toRelativeUrl(hostId, urlString, false);
                if (relativeUrl != null) {
                    importantRelUrls.add(relativeUrl);
                } else {
                    invalidUrls.add(urlString);
                }
            } catch (Exception e) {
                invalidUrls.add(urlString);
            }
        }
        List<ImportantUrlRequest> importantUrls = importantRelUrls.stream()
                .map(relUrl -> new ImportantUrlRequest(hostId, relUrl, DateTime.now()))
                .collect(Collectors.toList());
        // подсчитаем, сколько в итоге получится подписок
        // выкинем уже имеющиеся
        Set<String> existingUrls = importantUrlsService.listImportantUrlRequests(hostId)
                .stream()
                .map(ImportantUrlRequest::getRelativeUrl)
                .collect(Collectors.toSet());

        importantUrls = importantUrls.stream()
                .filter(url -> !existingUrls.contains(url.getRelativeUrl()))
                .collect(Collectors.toList());

        int quotaUsed = existingUrls.size() + importantUrls.size();
        int quotaRemain = importantUrlsService.getRecommendedUrlsLimit(request.getHostId()) - quotaUsed;
        // записываем в БД
        if (quotaRemain < 0) {
            return new AddImportantUrlsResponse.QuotaExceededResponse(importantUrlsService.getRecommendedUrlsLimit(request.getHostId()), quotaUsed,
                    quotaRemain, importantUrls.size());
        } else {
            importantUrlsService.addImportantUrls(importantUrls);
        }
        WarningType warning = invalidUrls.isEmpty() ? null : WarningType.ADD_IMPORTANT_URLS__INVALID_URLS;
        return new AddImportantUrlsResponse.NormalResponse(importantUrlsService.getRecommendedUrlsLimit(request.getHostId()), quotaUsed, quotaRemain,
                importantUrls.stream().map(ImportantUrl::createFromRequest).collect(Collectors.toList()),
                invalidUrls, warning);
    }

    @Required
    public void setImportantUrlsService(ImportantUrlsService importantUrlsService) {
        this.importantUrlsService = importantUrlsService;
    }
}
