package ru.yandex.webmaster3.viewer.http.indexing;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.link.DatePoint;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.cache.DashboardCacheService;
import ru.yandex.webmaster3.storage.cache.DashboardType;
import ru.yandex.webmaster3.storage.iks.IksService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.indexing.GetIksHistoryResponse.IksHistory;

/**
 * Created by ifilippov5 on 21.04.18.
 */
@ReadAction
@Category("indexing")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class GetIksHistoryDashboardAction extends AbstractUserVerifiedHostAction<GetIksHistoryRequest, GetIksHistoryResponse> {
    private static final Logger log = LoggerFactory.getLogger(GetIksHistoryDashboardAction.class);
    private static final DashboardType DASHBOARD_TYPE = DashboardType.INDEXING_IKS_HISTORY;

    private final DashboardCacheService dashboardCacheService;
    private final IksService iksService;

    private static String createKey(LocalDate lastUpdateDate) {
        return lastUpdateDate.toString() + "_v2";
    }

    @Override
    public GetIksHistoryResponse process(GetIksHistoryRequest request) {
        String owner = iksService.getMascotOwner(request.getHostId());
        WebmasterHostId ownerHostId = IdUtils.urlToHostId(owner);

        Pair<Boolean, String> ownerInfo = IksService.comparePunycodeHostNameWithPunycodeOwnerName(request.getHostId(), ownerHostId.getPunycodeHostname());
        Optional<DateTime> optLastUpdateCurrentIks = iksService.getCurrentIksLastImportDate();
        if (optLastUpdateCurrentIks.isEmpty()) {
            return new GetIksHistoryResponse.NormalResponse(ownerInfo.getLeft(), ownerInfo.getRight());
        }
        DateTime lastUpdateCurrentIks = optLastUpdateCurrentIks.get();

        String cacheKey = createKey(lastUpdateCurrentIks.toLocalDate());
        Optional<GetIksHistoryResponse> response = dashboardCacheService.getDataUnchecked(ownerHostId, DASHBOARD_TYPE, cacheKey,
                (valueBytes) -> JsonMapping.readValue(valueBytes, GetIksHistoryResponse.NormalResponse.class));
        if (response.isPresent()) {
            return response.get();
        }

        DateTime dateTo = request.getDateTo();
        if (dateTo == null || dateTo.isAfter(lastUpdateCurrentIks)) {
            //если текущий икс отстает от истории
            dateTo = lastUpdateCurrentIks;
        }
        List<DatePoint> history = iksService.getIksHistory(Collections.singleton(owner),
                request.getDateFrom() == null ? null : request.getDateFrom().toLocalDate(), dateTo.toLocalDate()).get(owner);

        var result = new GetIksHistoryResponse.NormalResponse(ownerInfo.getLeft(), ownerInfo.getRight(), new IksHistory(history));
        dashboardCacheService.saveData(ownerHostId, DASHBOARD_TYPE, cacheKey, JsonMapping.writeValueAsBytes(result));
        return result;
    }
}
