package ru.yandex.webmaster3.viewer.http.indexing2.download;

import com.google.common.base.Charsets;
import com.google.common.hash.Funnel;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskData;
import ru.yandex.webmaster3.storage.indexing2.download.DownloadIndexingUrlEventSamplesTaskData;
import ru.yandex.webmaster3.storage.indexing2.samples.IndexedUrlEventSampleFilter;
import ru.yandex.webmaster3.storage.indexing2.samples.IndexingSamplesService;
import ru.yandex.webmaster3.storage.indexing2.samples.dao.IndexingEventConditions;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.viewer.http.download.DownloadAction;
import ru.yandex.webmaster3.viewer.http.indexing2.events.IndexedUrlEventSamplesAction;

/**
 * Created by ifilippov5 on 28.02.17.
 */
@ReadAction
@Category("indexing")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class DownloadIndexedUrlEventSamplesAction extends DownloadAction<DownloadIndexedUrlEventsSamplesRequest> {

    private final IndexingSamplesService indexingSamplesService;

    @Override
    protected WorkerTaskData getTaskData(DownloadIndexedUrlEventsSamplesRequest request, long hash) {
        String fileName = generateFileName(request, hash);
        Condition condition = IndexedUrlEventSamplesAction.makeCondition(request.getParameterFilters())
                .andThen(getNodeCondition(request.getNodeId(), request.getHostId(), IndexingEventConditions::buildPathCondition));

        return DownloadIndexingUrlEventSamplesTaskData.createTaskData(
                request.getHostId(),
                condition,
                request.getOrderBy(),
                request.getOrderDirection(),
                request.getExportFormat(),
                fileName,
                hash
        );
    }

    @Override
    protected long getHash(DownloadIndexedUrlEventsSamplesRequest request) {
        HashFunction hf = Hashing.md5();

        Hasher hc = hf.newHasher()
                .putInt(request.getHostId().hashCode())
                .putString(request.getExportFormat().toString(), Charsets.UTF_8)
                .putString("event", Charsets.UTF_8)
                .putString(indexingSamplesService.getEventSamplesDataVersion(request.getHostId()), Charsets.UTF_8);

        if (request.getNodeId() != null)
            hc.putLong(request.getNodeId());

        if (request.getParameterFilters() != null) {
            for (IndexedUrlEventSampleFilter filter : request.getParameterFilters()) {
                hc.putObject(filter, (Funnel<IndexedUrlEventSampleFilter>) (searchUrlFilter, into) -> {
                    into.putString(searchUrlFilter.getOperation().toString(), Charsets.UTF_8);
                    into.putString(searchUrlFilter.getIndicator().toString(), Charsets.UTF_8);
                    into.putString(searchUrlFilter.getValue(), Charsets.UTF_8);
                });
            }
        }

        return hc.hash().asLong();
    }

}
