package ru.yandex.webmaster3.viewer.http.indexing2.download;

import com.google.common.base.Charsets;
import com.google.common.hash.Funnel;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskData;
import ru.yandex.webmaster3.storage.indexing2.download.DownloadIndexingUrlSamplesTaskData;
import ru.yandex.webmaster3.storage.indexing2.samples.IndexedUrlSampleFilter;
import ru.yandex.webmaster3.storage.indexing2.samples.IndexingSamplesService;
import ru.yandex.webmaster3.storage.indexing2.samples.dao.IndexedUrlSampleConditions;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.viewer.http.download.DownloadAction;
import ru.yandex.webmaster3.viewer.http.indexing2.urls.IndexedUrlSamplesAction;

/**
 * Created by ifilippov5 on 28.02.17.
 */
@ReadAction
@Category("indexing")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class DownloadIndexedUrlSamplesAction extends DownloadAction<DownloadIndexedUrlSamplesRequest> {

    private final IndexingSamplesService indexingSamplesService;

    @Override
    protected WorkerTaskData getTaskData(DownloadIndexedUrlSamplesRequest request, long hash) {
        String fileName = generateFileName(request, hash);
        Condition condition = IndexedUrlSamplesAction.makeCondition(request.getParameterFilters())
                .andThen(getNodeCondition(request.getNodeId(), request.getHostId(), IndexedUrlSampleConditions::buildPathCondition));

        return DownloadIndexingUrlSamplesTaskData.createTaskData(
                request.getHostId(),
                condition,
                request.getOrderBy(),
                request.getOrderDirection(),
                request.getExportFormat(),
                fileName,
                hash
        );
    }

    @Override
    protected long getHash(DownloadIndexedUrlSamplesRequest request) {
        HashFunction hf = Hashing.md5();

        Hasher hc = hf.newHasher()
                .putInt(request.getHostId().hashCode())
                .putString(request.getExportFormat().toString(), Charsets.UTF_8)
                .putString("url", Charsets.UTF_8)
                .putString(indexingSamplesService.getUrlSamplesDataVersion(request.getHostId()), Charsets.UTF_8);

        if (request.getNodeId() != null)
            hc.putLong(request.getNodeId());

        if (request.getParameterFilters() != null) {
            for (IndexedUrlSampleFilter filter : request.getParameterFilters()) {
                hc.putObject(filter, (Funnel<IndexedUrlSampleFilter>) (searchUrlFilter, into) -> {
                    into.putString(searchUrlFilter.getOperation().toString(), Charsets.UTF_8);
                    into.putString(searchUrlFilter.getIndicator().toString(), Charsets.UTF_8);
                    into.putString(searchUrlFilter.getValue(), Charsets.UTF_8);
                });
            }
        }

        return hc.hash().asLong();
    }
}
