package ru.yandex.webmaster3.viewer.http.indexing2.events;

import java.util.Collections;
import java.util.List;

import lombok.Setter;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.codes.HttpCodeGroup;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.AbstractFilter;
import ru.yandex.webmaster3.storage.TextFilterUtil;
import ru.yandex.webmaster3.storage.indexing2.samples.IndexingSamplesService;
import ru.yandex.webmaster3.storage.indexing2.samples.dao.IndexedUrlEventSampleField;
import ru.yandex.webmaster3.storage.indexing2.samples.dao.IndexingEventConditions;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Operator;
import ru.yandex.webmaster3.viewer.http.AbstractUrlFilteringAction;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * Created by leonidrom on 17/02/2017.
 */
@ReadAction
@Category("indexing")
public class IndexedUrlEventSamplesAction extends AbstractUrlFilteringAction<IndexedUrlEventSamplesRequest, IndexedUrlEventSamplesResponse> {

    @Setter
    private IndexingSamplesService indexingSamplesService;

    @Override
    public IndexedUrlEventSamplesResponse process(IndexedUrlEventSamplesRequest request)
            throws WebmasterException {
        Condition condition = makeCondition(request.getParameterFilters())
                .andThen(getNodeCondition(request.getNodeId(), request.getHostId(), IndexingEventConditions::buildPathCondition));
        long count = indexingSamplesService.getEventSamplesCount(request.getHostId(), condition);
        if (count == 0L) {
            return new IndexedUrlEventSamplesResponse(0, Collections.emptyList());
        }
        return new IndexedUrlEventSamplesResponse(
                count,
                indexingSamplesService.getEventSamples(
                        request.getHostId(),
                        condition,
                        request.getOrderBy(),
                        request.getOrderDirection(),
                        request.getPage() * request.getPageSize(),
                        request.getPageSize()
                )
        );
    }

    public static Condition makeCondition(List<? extends AbstractFilter<IndexedUrlEventSampleField>> filters) {
        Condition condition = Condition.trueCondition();
        if (filters != null) {
            for (AbstractFilter<IndexedUrlEventSampleField> filter : filters) {
                Operator dateOp = Operator.fromFilterOperation(filter.getOperation());

                switch (filter.getIndicator()) {
                    case LAST_ACCESS_DATE:
                        if (dateOp == null) {
                            throw filter.invalidFilterException();
                        } else {
                            condition = condition.andThen(IndexingEventConditions.lastAccessCondition(dateOp, filter.parseDate()));
                        }
                        break;
                    case URL:
                        condition = condition.andThen(TextFilterUtil.getTextCondition(filter, IndexingEventConditions.getUrlFieldName()));
                        break;
                    case CURRENT_CODE:
                        if (filter.getOperation() == AbstractFilter.Operation.EQUAL) {
                            // TODO: for backward compatibility until frontend migrates to the new filters
                            YandexHttpStatus yandexHttpStatus = parseStatusAsYandexHttpStatus(filter);
                            if (yandexHttpStatus != null) {
                                condition = condition.andThen(IndexingEventConditions.curCodeEqual(yandexHttpStatus));
                            } else {
                                condition = condition.andThen(IndexingEventConditions.curCodeEqual(parseStatus(filter)));
                            }
                        } else {
                            throw filter.invalidFilterException();
                        }
                        break;
                    case PREVIOUS_CODE:
                        if (filter.getOperation() == AbstractFilter.Operation.EQUAL) {
                            // TODO: for backward compatibility until frontend migrates to the new filters
                            YandexHttpStatus yandexHttpStatus = parseStatusAsYandexHttpStatus(filter);
                            if (yandexHttpStatus != null) {
                                condition = condition.andThen(IndexingEventConditions.prevCodeEqual(yandexHttpStatus));
                            } else {
                                condition = condition.andThen(IndexingEventConditions.prevCodeEqual(parseStatus(filter)));
                            }
                        } else {
                            throw filter.invalidFilterException();
                        }
                        break;
                    default:
                        throw new RuntimeException("Not implemented filter for indicator " + filter.getIndicator());
                }
            }
        }
        return condition;
    }

    private static HttpCodeGroup parseStatus(AbstractFilter<IndexedUrlEventSampleField> filter) {
        return HttpCodeGroup.valueOf(filter.getValue());
    }

    // TODO: for backward compatibility until frontend migrates to the new filters
    private static YandexHttpStatus parseStatusAsYandexHttpStatus(AbstractFilter<IndexedUrlEventSampleField> filter) {
        try {
            return YandexHttpStatus.valueOf(filter.getValue());
        } catch (Throwable t) {
            return null;
        }
    }

}
