package ru.yandex.webmaster3.viewer.http.indexing2.events;

import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.proto.dashboard.cache.DashboardCache;
import ru.yandex.webmaster3.storage.cache.DashboardType;
import ru.yandex.webmaster3.storage.indexing2.samples.IndexingSamplesService;
import ru.yandex.webmaster3.storage.indexing2.samples.data.IndexedUrlEventSample;
import ru.yandex.webmaster3.viewer.http.AbstractDashboardAction;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author avhaliullin
 */
@ReadAction
@Category("indexing")
public class IndexedUrlEventSamplesDashboardAction extends AbstractDashboardAction<IndexedUrlEventSamplesRequest, IndexedUrlEventSamplesResponse, IndexedUrlEventSamplesAction> {
    private IndexingSamplesService indexingSamplesService;

    public IndexedUrlEventSamplesDashboardAction() {
        super(DashboardType.INDEXING_EVENT_SAMPLES_HISTORY);
    }

    @Override
    protected byte[] serializeResponse(IndexedUrlEventSamplesResponse response) {
        return DashboardCache.IndexingEventSamplesCache.newBuilder()
                .setTotalCount(response.getTotalSamples())
                .addAllSamples(
                        response.getSamples().stream()
                                .map(sample ->
                                        DashboardCache.IndexingEventSamplesCache.IndexingEventSample.newBuilder()
                                                .setCurrentCode(sample.getCurrentCode().getCode())
                                                .setPreviousCode(sample.getPreviousCode().getCode())
                                                .setLastAccess(sample.getLastAccess().getMillis())
                                                .setUrl(sample.getUrl())
                                                .build()
                                ).collect(Collectors.toList())
                ).build().toByteArray();
    }

    @Override
    protected Optional<IndexedUrlEventSamplesResponse> deserializeResponse(byte[] data) throws IOException {
        DashboardCache.IndexingEventSamplesCache cacheProto = DashboardCache.IndexingEventSamplesCache.parseFrom(data);
        long totalCount = cacheProto.getTotalCount();
        List<IndexedUrlEventSample> samples = cacheProto.getSamplesList().stream()
                .map(sample -> new IndexedUrlEventSample(
                        sample.getUrl(),
                        HttpCodeInfo.fromHttpStatus(YandexHttpStatus.parseCode(sample.getPreviousCode())),
                        HttpCodeInfo.fromHttpStatus(YandexHttpStatus.parseCode(sample.getCurrentCode())),
                        new DateTime(sample.getLastAccess())
                )).collect(Collectors.toList());
        return Optional.of(new IndexedUrlEventSamplesResponse(totalCount, samples));
    }

    @Override
    protected String buildCacheKey(IndexedUrlEventSamplesRequest request) {
        return indexingSamplesService.getEventSamplesDataVersion(request.getHostId());
    }

    @Required
    public void setIndexingSamplesService(IndexingSamplesService indexingSamplesService) {
        this.indexingSamplesService = indexingSamplesService;
    }
}
