package ru.yandex.webmaster3.viewer.http.indexing2.urls;

import java.io.IOException;
import java.util.Optional;
import java.util.stream.Collectors;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.proto.dashboard.cache.DashboardCache;
import ru.yandex.webmaster3.storage.cache.DashboardType;
import ru.yandex.webmaster3.storage.indexing2.history.IndexedUrlsCountHistoryService;
import ru.yandex.webmaster3.viewer.http.AbstractDashboardAction;
import ru.yandex.webmaster3.viewer.http.indexing2.urls.data.IndexedUrlStatusGroup;
import ru.yandex.webmaster3.viewer.util.DashboardUtils;

/**
 * @author avhaliullin
 */
@ReadAction
@Category("indexing")
public class IndexingUrlsCountHistoryDashboardAction extends AbstractDashboardAction<IndexedUrlsCountHistoryRequest, IndexedUrlsCountHistoryResponse, IndexedUrlsCountHistoryAction> {

    private static final Duration DATE_RANGE = Duration.standardDays(180);

    private IndexedUrlsCountHistoryService indexedUrlsCountHistoryService;

    public IndexingUrlsCountHistoryDashboardAction() {
        super(DashboardType.INDEXING_URLS_COUNT_HISTORY);
    }

    @Override
    protected byte[] serializeResponse(IndexedUrlsCountHistoryResponse response) {
        return DashboardCache.IndexingUrlsCountHistoryCache.newBuilder()
                .addAllHistories(response.getHistories().stream()
                        .map(info -> DashboardCache.IndexingUrlsCountHistoryCache.IndexingUrlsCountHistory.newBuilder()
                                .addAllData(DashboardUtils.datePointsToProto(info.getData()))
                                .setIndicator(info.getIndicatorName().value())
                                .build()
                        ).collect(Collectors.toList())
                ).build()
                .toByteArray();
    }

    @Override
    protected Optional<IndexedUrlsCountHistoryResponse> deserializeResponse(byte[] data) throws IOException {
        DashboardCache.IndexingUrlsCountHistoryCache protoMessage =
                DashboardCache.IndexingUrlsCountHistoryCache.parseFrom(data);
        return Optional.of(new IndexedUrlsCountHistoryResponse(
                protoMessage.getHistoriesList()
                        .stream()
                        .map(info -> new IndexedUrlsCountHistoryResponse.IndexedUrlsHistory(
                                DashboardUtils.datePointsFromProto(info.getDataList()),
                                IndexedUrlStatusGroup.R.fromValue(info.getIndicator())
                        )).collect(Collectors.toList())
        ));
    }

    @Override
    protected void prepareRequest(IndexedUrlsCountHistoryRequest request) {
        // set date range
        if (request.getDateTo() == null) {
            request.setDateTo(Instant.now());
        }
        if (request.getDateFrom() == null) {
            request.setDateFrom(request.getDateTo().minus(DATE_RANGE));
        }
        super.prepareRequest(request);
    }

    @Override
    protected String buildCacheKey(IndexedUrlsCountHistoryRequest request) {
        return indexedUrlsCountHistoryService.getUrlHistoryDataVersion(request.getHostId());
    }

    @Required
    public void setIndexedUrlsCountHistoryService(IndexedUrlsCountHistoryService indexedUrlsCountHistoryService) {
        this.indexedUrlsCountHistoryService = indexedUrlsCountHistoryService;
    }
}
