package ru.yandex.webmaster3.viewer.http.links;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.core.codes.ErrorGroupEnum;
import ru.yandex.webmaster3.core.codes.LinkType;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.link.LinkHistoryIndicatorType;
import ru.yandex.webmaster3.core.link.LinksHistoryIndicator;
import ru.yandex.webmaster3.storage.links.LinksService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * links (ALL)
 * |- external links (EXTERNAL)
 * |- internal links (INTERNAL) - normal/broken
 *    |- normal links (NORMAL_HTTP_CODES) - http codes
 *    |- broken links (BROKEN_GROUPS) - disallowed by user/site errors/unsupported by robot
 *       |- disallowed by user (BROKEN__SITE_ERROR) - http codes
 *       |- site errors (BROKEN__DISALLOWED_BY_USER)
 *       |- unsupported by robot (BROKEN__UNSUPPORTED_BY_ROBOT)
 *
 * @author aherman
 */
@ReadAction
@Category("links")
public class GetLinkIndicatorsTreeAction extends AbstractUserVerifiedHostAction<GetLinkIndicatorsTreeRequest, GetLinkIndicatorsTreeResponse> {
    private LinksService linksService;

    @Override
    public GetLinkIndicatorsTreeResponse process(GetLinkIndicatorsTreeRequest request) {
        List<LinksHistoryIndicator> availableIndicators =
                linksService.getActualIndicators(request.getHostId());

        Set<Integer> httpCodes = availableIndicators.stream()
                .filter(i -> i.getType() == LinkHistoryIndicatorType.INTERNAL_LINKS_HTTP_CODES)
                .map(LinksHistoryIndicator::getHttpCode)
                .collect(Collectors.toSet());

        Set<LinkIndicator.All> indicators = new HashSet<>();
        indicators.add(LinkIndicator.All.LINKS);
        if (availableIndicators.stream().filter(i -> i.getType() == LinkHistoryIndicatorType.EXTERNAL_LINK_URLS_COUNT).findAny().isPresent()) {
            indicators.add(LinkIndicator.All.LINKS_EXTERNAL);
        }

        for (Integer availableHttpCode : httpCodes) {
            if (availableHttpCode == null) {
                continue;
            }

            LinkType linkType = LinkType.get(availableHttpCode);

            if (linkType == LinkType.NORMAL) { //DownloadedHttpCodeGroup.get(availableHttpCode).isPresent()) {
                indicators.add(LinkIndicator.All.LINKS_INTERNAL);
                indicators.add(LinkIndicator.All.LINKS_INTERNAL_NORMAL);
            } else if (linkType == LinkType.REDIRECT) {
                indicators.add(LinkIndicator.All.LINKS_INTERNAL);
                indicators.add(LinkIndicator.All.LINKS_INTERNAL_REDIRECT);
            } else if (linkType == LinkType.NOT_DOWNLOADED) {
                indicators.add(LinkIndicator.All.LINKS_INTERNAL);
                indicators.add(LinkIndicator.All.LINKS_INTERNAL_NOT_DOWNLOADED);
            } else if (linkType == LinkType.BROKEN) {
                ErrorGroupEnum errorGroupEnum = ErrorGroupEnum.getForced(availableHttpCode);
                indicators.add(LinkIndicator.All.LINKS_INTERNAL);
                indicators.add(LinkIndicator.All.LINKS_INTERNAL_BROKEN);
                indicators.add(LinkIndicator.brokenToAll(LinkIndicator.fromGroup(errorGroupEnum)));
            }
        }

        List<GetLinkIndicatorsTreeResponse.Node> nodes =
                // Persisting order
                Arrays.stream(LinkIndicator.All.values())
                        .filter(indicators::contains)
                        .map(GetLinkIndicatorsTreeResponse.Node::new)
                        .collect(Collectors.toList());
        return new GetLinkIndicatorsTreeResponse.NormalResponse(nodes);
    }

    @Required
    public void setLinksService(LinksService linksService) {
        this.linksService = linksService;
    }
}
