package ru.yandex.webmaster3.viewer.http.links;

import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;

import com.google.common.collect.Lists;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.link.LinkHistoryIndicatorType;
import ru.yandex.webmaster3.storage.links.LinksService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author azakharov
 */
@ReadAction
@Description("Ручка для графиков истории по количеству внутренних и внешних ссылок")
@Category("links")
public class GetLinksHistoryAction extends AbstractUserVerifiedHostAction<GetLinksHistoryRequest, GetLinksHistoryResponse> {
    private LinksService linksService;

    @Override
    public GetLinksHistoryResponse process(GetLinksHistoryRequest request) {
        Map<LinkHistoryIndicatorType, NavigableMap<LocalDate, Long>> simpleHistory =
                linksService.getSimpleHistory(request.getHostId());
        NavigableMap<LocalDate, Long> externalUrlsCount = simpleHistory.computeIfAbsent(
                LinkHistoryIndicatorType.EXTERNAL_LINK_URLS_COUNT,
                k -> new TreeMap<>()
        );
        NavigableMap<LocalDate, Long> internalUrlsCount = simpleHistory.computeIfAbsent(
                LinkHistoryIndicatorType.INTERNAL_LINKS_URL_COUNT,
                k -> new TreeMap<>()
        );

        return new GetLinksHistoryResponse.OrdinaryResponse(Lists.newArrayList(
                new GetLinksHistoryResponse.LinkHistory(
                        GetLinksHistoryResponse.Indicator.LINKS_EXTERNAL, HistoryUtil.toDataPoint(externalUrlsCount)
                ),
                new GetLinksHistoryResponse.LinkHistory(
                        GetLinksHistoryResponse.Indicator.LINKS_INTERNAL, HistoryUtil.toDataPoint(internalUrlsCount)
                )
        ));
    }

    @Required
    public void setLinksService(LinksService linksService) {
        this.linksService = linksService;
    }
}
