package ru.yandex.webmaster3.viewer.http.links2;

import com.google.protobuf.InvalidProtocolBufferException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.link.HostLinkSample;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.proto.dashboard.cache.DashboardCache;
import ru.yandex.webmaster3.storage.cache.DashboardCacheService;
import ru.yandex.webmaster3.storage.cache.DashboardType;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.viewer.http.links.LinkSamplesResponse;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * @author aherman
 */
@ReadAction
@Category("links")
public class GetExternalLinkSamplesDashboardAction extends GetExternalLinkSamplesAction {
    private static final Logger log = LoggerFactory.getLogger(GetExternalLinkSamplesDashboardAction.class);
    private static final DashboardType DASHBOARD_TYPE = DashboardType.LINK_EXTERNAL_SAMPLES;

    private DashboardCacheService dashboardCacheService;

    @Override
    public LinkSamplesResponse process(GetExternalLinkSamplesRequest request) {
        String cacheKey = createKey();

        try {
            Optional<List<HostLinkSample>> cachedSampls =
                    dashboardCacheService.getData(request.getHostId(), DASHBOARD_TYPE, cacheKey, (valueBytes) -> {
                        DashboardCache.LinkExternalSamplesCache lesc;
                        try {
                            lesc = DashboardCache.LinkExternalSamplesCache.parseFrom(valueBytes);
                        } catch (InvalidProtocolBufferException e) {
                            log.error("Unable to read value", e);
                            return Optional.empty();
                        }
                        List<HostLinkSample> result = new ArrayList<>();
                        for (int i = 0; i < lesc.getSamplesCount(); i++) {
                            DashboardCache.LinkExternalSamplesCache.LinkSample sample = lesc.getSamples(i);
                            result.add(new HostLinkSample(
                                    IdUtils.stringToHostId(sample.getHostId()),
                                    sample.getHostUrl(),
                                    sample.getHostPath(),
                                    TimeUtils.unixTimestampToInstant(sample.getHostUrlLastAccessDate()),
                                    sample.getHostHttpCode(),
                                    new HttpCodeInfo(sample.getHostHttpCode()),
                                    null,
                                    null,
                                    sample.getSourceUrl(),
                                    TimeUtils.unixTimestampToInstant(sample.getSourceUrlLastAccessDate()),
                                    sample.getSourceHttpCode(),
                                    new HttpCodeInfo(sample.getSourceHttpCode()),
                                    0, //TODO !!!!!!!!!!
                                    TimeUtils.unixTimestampToInstant(sample.getLinkDiscoverDate()),
                                    null,
                                    null
                            ));
                        }
                        return Optional.of(result);
                    });
            if (cachedSampls.isPresent()) {
                log.debug("From cache: {} {} {}", request.getHostId(), DASHBOARD_TYPE, cacheKey);
                return new LinkSamplesResponse.NormalResponse(cachedSampls.get(), cachedSampls.get().size());
            }
        } catch (Exception e) {
            log.error("Unable to read cached value: {} {} {}", request.getHostId(), DASHBOARD_TYPE, cacheKey, e);
        }

        List<HostLinkSample> samples = link2SamplesService.listExternalSamples(request.getHostId(),
                0, request.getPageSize(),
                Condition.trueCondition(),
                request.getFilterBroken(),
                request.getOrderBy(), request.getOrderDirection(), request.isDeleted());
        try {
            DashboardCache.LinkExternalSamplesCache.Builder lesc = DashboardCache.LinkExternalSamplesCache.newBuilder();
            for (HostLinkSample sample : samples) {
                lesc.addSamples(DashboardCache.LinkExternalSamplesCache.LinkSample.newBuilder()
                        .setHostId(sample.getHostId().toStringId())
                        .setHostUrl(sample.getHostUrl())
                        .setHostPath(sample.getHostPath())
                        .setHostUrlLastAccessDate(TimeUtils.toUnixTimestamp(sample.getHostUrlLastAccessDate()))
                        .setHostHttpCode(sample.getHostHttpCode())
                        .setSourceUrl(sample.getSourceUrl())
                        .setSourceHttpCode(sample.getSourceHttpCode())
                        .setLinkDiscoverDate(TimeUtils.toUnixTimestamp(sample.getLinkDiscoverDate()))
                );
            }
            dashboardCacheService.saveData(request.getHostId(), DASHBOARD_TYPE, cacheKey, lesc.build().toByteArray());
        } catch (Exception e) {
            log.error("Unable to update cached value: {} {} {}", request.getHostId(), DASHBOARD_TYPE, cacheKey, e);
        }

        return new LinkSamplesResponse.NormalResponse(samples, samples.size());
    }

    private String createKey() {
        return link2SamplesService.getLinksTableName();
    }

    @Required
    public void setDashboardCacheService(DashboardCacheService dashboardCacheService) {
        this.dashboardCacheService = dashboardCacheService;
    }
}
