package ru.yandex.webmaster3.viewer.http.links2;

import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.format.ISODateTimeFormat;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.TextFilterUtil;
import ru.yandex.webmaster3.storage.links2.dao.*;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.LongCondition;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Operator;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.TimestampCondition;

import java.util.ArrayList;
import java.util.List;

/**
 * @author aherman
 */
class LinkConditionConverter {
    @NotNull
    static Condition convertLinkCondition(List<ViewerExtLinkFilter> parameterFilters, boolean deleted) {
        List<Condition> conditions = new ArrayList<>();
        for (ViewerExtLinkFilter parameterFilter : parameterFilters) {
            ViewerExtLinkFilter.Operation operation = parameterFilter.getOperation();
            Operator conditionOperator = null;
            switch (operation) {
                case GREATER_EQUAL:
                    conditionOperator = Operator.GREATER_EQUAL;
                    break;
                case GREATER_THAN:
                    conditionOperator = Operator.GREATER_THAN;
                    break;
                case LESS_EQUAL:
                    conditionOperator = Operator.LESS_EQUAL;
                    break;
                case LESS_THAN:
                    conditionOperator = Operator.LESS_THAN;
                    break;
            }

            switch (parameterFilter.getIndicator()) {
                case SOURCE_URL:
                    conditions.add(TextFilterUtil.getTextCondition(parameterFilter, ExternalLinks2SamplesCHDao.F.SOURCE_URL_UTF.getName()));
                    break;
                case HOST_URL:
                    conditions.add(TextFilterUtil.getTextCondition(parameterFilter, ExternalLinks2SamplesCHDao.F.HOST_URL_UTF.getName()));
                    break;

                case LINK_DISCOVERY_DATE: {
                    if (conditionOperator == null) {
                        throw new WebmasterException("Unknown operation",
                                new WebmasterErrorResponse.IllegalParameterValueResponse(LinkConditionConverter.class, null, null));
                    }
                    DateTime dateTimeValue = parseDate(parameterFilter);
                    conditions.add(new TimestampCondition(ExternalLinks2SamplesCHDao.F.LINK_DATE.getName(), conditionOperator, dateTimeValue));
                }
                break;

                case SOURCE_SQI: {
                    if (conditionOperator == null) {
                        throw new WebmasterException("Unknown operation",
                                new WebmasterErrorResponse.IllegalParameterValueResponse(LinkConditionConverter.class, null, null));
                    }
                    long longValue = parseLong(parameterFilter);
                    conditions.add(new LongCondition(ExternalLinks2SamplesCHDao.F.SOURCE_IKS.getName(), conditionOperator, longValue));
                }
                break;

                case SOURCE_LAST_ACCESS_DATE: {
                    if (conditionOperator == null) {
                        throw new WebmasterException("Unknown operation",
                                new WebmasterErrorResponse.IllegalParameterValueResponse(LinkConditionConverter.class, null, null));
                    }
                    DateTime dateTimeValue = parseDate(parameterFilter);
                    conditions.add(new TimestampCondition(ExternalLinks2SamplesCHDao.F.SOURCE_LAST_ACCESS.getName(), conditionOperator, dateTimeValue));
                }
                break;

                case HOST_LAST_ACCESS_DATE: {
                    if (conditionOperator == null) {
                        throw new WebmasterException("Unknown operation",
                                new WebmasterErrorResponse.IllegalParameterValueResponse(LinkConditionConverter.class, null, null));
                    }
                    DateTime dateTimeValue = parseDate(parameterFilter);
                    conditions.add(new TimestampCondition(ExternalLinks2SamplesCHDao.F.HOST_LAST_ACCESS.getName(), conditionOperator, dateTimeValue));
                }
                break;

                case DELETION_DATE: {
                    if (!deleted) {
                        continue;
                    }
                    if (conditionOperator == null) {
                        throw new WebmasterException("Unknown operation",
                                new WebmasterErrorResponse.IllegalParameterValueResponse(
                                        LinkConditionConverter.class, null, null));
                    }
                    DateTime dateTimeValue = parseDate(parameterFilter);
                    conditions.add(new TimestampCondition(ExternalLinks2SamplesCHDao.F.DELETED_DATE.getName(), conditionOperator, dateTimeValue));
                }
            }
        }

        return Condition.and(conditions);
    }

    private static DateTime parseDate(ViewerExtLinkFilter parameterFilter) {
        try {
            return ISODateTimeFormat.dateTime().parseDateTime(parameterFilter.getValue());
        } catch (IllegalArgumentException e) {
            throw new WebmasterException("Unable to parse date",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(LinkConditionConverter.class, null, null), e);
        }
    }

    private static long parseLong(ViewerExtLinkFilter parameterFilter) {
        try {
            return Long.parseLong(parameterFilter.getValue());
        } catch (IllegalArgumentException e) {
            throw new WebmasterException("Unable to parse long value",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(LinkConditionConverter.class, null, null), e);
        }
    }
}
