package ru.yandex.webmaster3.viewer.http.metrika.counter;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ActionStatus;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.request.PagerAware;
import ru.yandex.webmaster3.core.metrika.counters.CounterBindingStateEnum;
import ru.yandex.webmaster3.core.metrika.counters.MetrikaCounterBindingInfo;
import ru.yandex.webmaster3.storage.metrika.MetrikaCounterBindingService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

import java.util.EnumSet;
import java.util.List;

/**
 * Created by ifilippov5 on 27.11.17.
 */
@ReadAction
public class GetBindingsListAction extends Action<GetBindingsListAction.Request, GetBindingsListAction.Response> {
    private static final Logger log = LoggerFactory.getLogger(GetBindingsListAction.class);

    private final MetrikaCounterBindingService metrikaCounterBindingService;
    private BindingsListTypeEnum listType;

    @Autowired
    public GetBindingsListAction(MetrikaCounterBindingService metrikaCounterBindingService) {
        this.metrikaCounterBindingService = metrikaCounterBindingService;
    }

    @Override
    public Response process(Request request) throws WebmasterException {
        EnumSet<CounterBindingStateEnum> states;
        if (listType == BindingsListTypeEnum.METRIKA_REQUEST) {
            states = EnumSet.of(CounterBindingStateEnum.METRIKA_REQUEST);
        } else {
            states = EnumSet.of(CounterBindingStateEnum.APPROVED, CounterBindingStateEnum.METRIKA_REJECT,
                    CounterBindingStateEnum.WEBMASTER_REQUEST);
        }
        Pair<Long, List<MetrikaCounterBindingInfo>> bindingsInfo = metrikaCounterBindingService.getBindingsWithCounterInfo(request.getHostId(), states,
                request.getP() * request.getPSize(), request.getPSize());

        return new Response(bindingsInfo.getLeft(), bindingsInfo.getRight());
    }

    @Required
    public void setListType(BindingsListTypeEnum listType) {
        this.listType = listType;
    }

    public static class Request extends AbstractUserVerifiedHostRequest implements PagerAware {
        private int p;
        private int pSize = 5;

        public int getP() {
            return p;
        }

        @Override
        public void setP(int p) {
            this.p = p;
        }

        public int getPSize() {
            return pSize;
        }

        @Override
        public void setPSize(int pSize) {
            this.pSize = pSize;
        }
    }

    public class Response implements ActionResponse {
        private long bindingsCount;
        private List<MetrikaCounterBindingInfo> bindings;

        public Response(long bindingsCount, List<MetrikaCounterBindingInfo> bindings) {
            this.bindingsCount = bindingsCount;
            this.bindings = bindings;
        }

        @Description("количество привязок")
        public long getBindingsCount() {
            return bindingsCount;
        }

        @Description("список привязок с информацией о счетчике")
        public List<MetrikaCounterBindingInfo> getBindings() {
            return bindings;
        }

        @Override
        public ActionStatus getRequestStatus() {
            return ActionStatus.SUCCESS;
        }
    }
}
