package ru.yandex.webmaster3.viewer.http.microdata;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.viewer.http.AbstractUserAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserRequest;
import ru.yandex.webmaster3.viewer.microdata.semantic.SemanticValidation;

import java.util.Map;

@ReadAction
@Description("Заросить валидацию микроразметки по урлу документа")
@Category("validation")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Component("/microdata/semantic/byDocument")
public class MicrodataSemanticValidateByDocumentAction extends
        AbstractUserAction<MicrodataSemanticValidateByDocumentAction.Request, MicrodataSemanticValidateByDocumentAction.Response> {
    private final SemanticValidation semanticValidation;

    @Override
    public Response process(Request request) {
        String reqId = UUIDs.timeBased().toString();
        Map<String, JsonNode> value = semanticValidation.validateByDocument(request.getHtml(), request.getLang(), reqId, request.onlyError);
        return createResponse(value, reqId);
    }

    public static final Response.DataResponse createResponse(Map<String, JsonNode> jsonNode, String reqId) {
        return Response.DataResponse.builder().microdata(jsonNode.get("microdata"))
                .rdfa(jsonNode.get("rdfa"))
                .jsonLd(jsonNode.get("json-ld"))
                .microformat(jsonNode.get("microformat"))
                .id(reqId)
                .build();
    }

    @Data
    public static class Request extends AbstractUserRequest {

        @Setter(onMethod = @__({@Description("HTML документ для валидации"), @RequestQueryProperty(required = true)}))
        private String html;
        @Setter(onMethod = @__({@Description("Признак того что надо возврашать только ощибки"), @RequestQueryProperty(required = false)}))
        private boolean onlyError;
        @Setter(onMethod = @__({@Description("Язык"), @RequestQueryProperty(required = false)}))
        private String lang = "ru";

    }

    public static abstract class Response implements ActionResponse {
        @Value
        @Builder
        public static class DataResponse extends Response implements ActionResponse.NormalResponse {
            String id;
            JsonNode microdata;
            JsonNode rdfa;
            @JsonProperty("json-ld")
            JsonNode jsonLd;
            JsonNode microformat;
        }

    }
}
