package ru.yandex.webmaster3.viewer.http.mirrors;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.checklist.data.SummarySiteProblemsInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.service.DisplayNameService2;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.mirrors.dao.MirrorChangeDisallowedOwnersYDao;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorActionEnum;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequest;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequestStateEnum;
import ru.yandex.webmaster3.storage.mirrors.service.MainMirrorService;
import ru.yandex.webmaster3.storage.user.UserUnverifiedHost;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.common.response.CommonHostInfo;
import ru.yandex.webmaster3.viewer.http.user.GetHostListAction;

/**
 * User: azakharov
 * Date: 06.07.15
 * Time: 18:22
 */
@WriteAction
@Description("Ручка для сохранения заявки на изменение главного зеркала")
@Category("mirrors")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/mirrors/change")
public class ChangeMainMirrorAction extends AbstractUserVerifiedHostAction<ChangeMainMirrorRequest, ChangeMainMirrorResponse> {

    private final DisplayNameService2 displayNameService2;
    private final HostOwnerService hostOwnerService;
    private final MainMirrorService mainMirrorService;
    private final MirrorChangeDisallowedOwnersYDao mirrorChangeDisallowedOwnersYDao;
    private final MirrorService2 mirrorService2;
    private final SiteProblemsService siteProblemsService;
    private final UserHostsService userHostsService;

    @Override
    public ChangeMainMirrorResponse process(ChangeMainMirrorRequest request) {
        // check selected new main mirror is a user host or https/www clone of user host
        WebmasterHostId newMainMirrorId = request.getNewMainMirrorId();

        Set<WebmasterHostId> unverifiedHosts =
                userHostsService.getMayBeUnverifiedHosts(request.getWebmasterUser()).stream()
                        .map(UserUnverifiedHost::getWebmasterHostId)
                        .collect(Collectors.toSet());
        Set<WebmasterHostId> verifiedHosts =
                userHostsService.getVerifiedHosts(request.getWebmasterUser()).stream()
                        .map(UserVerifiedHost::getWebmasterHostId)
                        .collect(Collectors.toSet());

        List<WebmasterHostId> candidates = new ArrayList<>();
        // host as it cames in parameter newMainMirrorId
        candidates.add(newMainMirrorId);
        // http/https switched newMainMirrorId
        WebmasterHostId httpsMainMirrorId = IdUtils.switchHttps(newMainMirrorId);
        candidates.add(httpsMainMirrorId);
        // www!www switcher newMainMirrorId
        candidates.add(
                new WebmasterHostId(
                        newMainMirrorId.getSchema(),
                        WwwUtil.switchWWW(newMainMirrorId.getPunycodeHostname()),
                        newMainMirrorId.getPort()));
        // http/https and www!www switched newMainMirrorId
        candidates.add(
                new WebmasterHostId(
                        httpsMainMirrorId.getSchema(),
                        WwwUtil.switchWWW(httpsMainMirrorId.getPunycodeHostname()),
                        httpsMainMirrorId.getPort()));

        long found = candidates.stream().filter(c -> unverifiedHosts.contains(c) || verifiedHosts.contains(c)).count();
        if (found == 0L) {
            return new ChangeMainMirrorResponse.InvalidNewMainHost();
        }

        WebmasterHostId oldMainMirror = mirrorService2.getMainMirror(request.getHostId());
        if (oldMainMirror == null) {
            oldMainMirror = request.getHostId();
        }

        if (request.isCheckActionAvailability()) {
            if (MirroringActionsUtil.getAvailableAction(request.getHostId(), oldMainMirror) != MirrorActionEnum.MOVE) {
                return new ChangeMainMirrorResponse.UnsupportedAction();
            }
        }

        MirrorRequest currentMirrorRequest = mainMirrorService.getLatestRequest(request.getHostId());
        if (currentMirrorRequest != null) {
            if (currentMirrorRequest.getState() == MirrorRequestStateEnum.NEW) {
                return new ChangeMainMirrorResponse.PreviousTaskNotFinished();
            } else if (currentMirrorRequest.getState() == MirrorRequestStateEnum.WAITING) {
                if (currentMirrorRequest.getAction() == MirrorActionEnum.MOVE &&
                        Objects.equals(request.getNewMainMirrorId(), currentMirrorRequest.getNewMainMirrorHostId())) {
                    return new ChangeMainMirrorResponse.SameTaskWaiting();
                }
            }
        }

        MirrorChangeDisallowedOwnersYDao.Record rec = mirrorChangeDisallowedOwnersYDao.get(
                hostOwnerService.getHostOwner(request.getNewMainMirrorId()));
        if (rec != null) {
            return new ChangeMainMirrorResponse.ChangeDisallowed(rec.threat);
        }

        MirrorRequest newMirrorRequest = mainMirrorService.addMoveRequest(
                request.getHostId(),
                oldMainMirror,
                request.getNewMainMirrorId()
        );

        mainMirrorService.scheduleTask(request.getHostId());

        CommonHostInfo newMainMirrorHostDisplayInfo = null;
        CommonHostInfo oldMainMirrorHostDisplayInfo = null;

        //noinspection Duplicates
        if (newMirrorRequest.getNewMainMirrorHostId() != null) {
            SummarySiteProblemsInfo rtProblems = siteProblemsService.getRealTimeProblemsSummary(newMirrorRequest.getNewMainMirrorHostId());
            String displayName = displayNameService2.getHostUrlWithDisplayName(request.getHostId());
            newMainMirrorHostDisplayInfo = GetHostListAction.toCommonHostInfo(newMirrorRequest.getNewMainMirrorHostId(), displayName, rtProblems, false);
        }

        //noinspection Duplicates
        if (newMirrorRequest.getOldMainMirrorHostId() != null) {
            SummarySiteProblemsInfo rtProblems = siteProblemsService.getRealTimeProblemsSummary(newMirrorRequest.getOldMainMirrorHostId());
            String displayName = displayNameService2.getHostUrlWithDisplayName(request.getHostId());
            oldMainMirrorHostDisplayInfo = GetHostListAction.toCommonHostInfo(newMirrorRequest.getOldMainMirrorHostId(), displayName, rtProblems, false);
        }
        return new ChangeMainMirrorResponse.OrdinaryResponse(newMirrorRequest, newMainMirrorHostDisplayInfo, oldMainMirrorHostDisplayInfo);
    }
}
