package ru.yandex.webmaster3.viewer.http.mirrors;

import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.joda.time.Days;
import org.joda.time.ReadablePeriod;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.checklist.data.SummarySiteProblemsInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.service.DisplayNameService2;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorActionEnum;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequest;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequestStateEnum;
import ru.yandex.webmaster3.storage.mirrors.service.MainMirrorService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.common.response.CommonHostInfo;
import ru.yandex.webmaster3.viewer.http.user.GetHostListAction;

/**
 * User: azakharov
 * Date: 08.07.15
 * Time: 20:12
 */
@Description("Ручка для получения информации о заявках на изменение главного зеркала")
@ReadAction
@Category("mirrors")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/mirrors/requests")
public class GetMainMirrorChangeRequestsAction
        extends AbstractUserVerifiedHostAction<GetMainMirrorChangeRequestsRequest, GetMainMirrorChangeRequestsResponse> {
    private static final ReadablePeriod HIDE_OLD_REQUEST_PERIOD = Days.SEVEN;

    private final DisplayNameService2 displayNameService2;
    private final MainMirrorService mainMirrorService;
    private final MirrorService2 mirrorService2;
    private final SiteProblemsService siteProblemsService;


    public GetMainMirrorChangeRequestsResponse process(GetMainMirrorChangeRequestsRequest request) {
        WebmasterHostId oldMainMirror = mirrorService2.getMainMirror(request.getHostId());
        MirrorActionEnum availableAction = MirroringActionsUtil.getAvailableAction(request.getHostId(), oldMainMirror);

        MirrorRequest currentMirrorRequest = mainMirrorService.getLatestRequest(request.getHostId());
        if (currentMirrorRequest == null) {
            return new GetMainMirrorChangeRequestsResponse.OrdinaryResponse(null, null, null, null, availableAction);
        }

        MirrorRequestStateEnum state = currentMirrorRequest.getState();
        if (state == MirrorRequestStateEnum.SUCCESS
                || state == MirrorRequestStateEnum.EXPIRED
                || state == MirrorRequestStateEnum.SERVICE_ERROR
                || state == MirrorRequestStateEnum.USER_ERROR) {
            DateTime tooOld = DateTime.now().minus(HIDE_OLD_REQUEST_PERIOD);
            if (currentMirrorRequest.isHidden() || currentMirrorRequest.getCreateDate().isBefore(tooOld)) {
                // Hide old requests
                return new GetMainMirrorChangeRequestsResponse.OrdinaryResponse(null, null, null, null, availableAction);
            }
        }

        CommonHostInfo newMainMirrorHostInfo = null;
        CommonHostInfo oldMainMirrorHostInfo = null;

        //noinspection Duplicates
        if (currentMirrorRequest.getNewMainMirrorHostId() != null) {
            SummarySiteProblemsInfo rtProblems = siteProblemsService.getRealTimeProblemsSummary(currentMirrorRequest.getNewMainMirrorHostId());
            String displayName = displayNameService2.getHostUrlWithDisplayName(request.getHostId());
            newMainMirrorHostInfo = GetHostListAction.toCommonHostInfo(currentMirrorRequest.getNewMainMirrorHostId(), displayName, rtProblems, false);
        }

        //noinspection Duplicates
        if (currentMirrorRequest.getOldMainMirrorHostId() != null) {
            SummarySiteProblemsInfo rtProblems = siteProblemsService.getRealTimeProblemsSummary(currentMirrorRequest.getOldMainMirrorHostId());
            String displayName = displayNameService2.getHostUrlWithDisplayName(request.getHostId());
            oldMainMirrorHostInfo = GetHostListAction.toCommonHostInfo(currentMirrorRequest.getOldMainMirrorHostId(), displayName, rtProblems, false);
        }

        boolean finished = (currentMirrorRequest.getState() != MirrorRequestStateEnum.NEW);

        return new GetMainMirrorChangeRequestsResponse.OrdinaryResponse(currentMirrorRequest,
                newMainMirrorHostInfo,
                oldMainMirrorHostInfo,
                finished, availableAction);
    }
}
