package ru.yandex.webmaster3.viewer.http.mobile;

import org.springframework.beans.factory.annotation.Required;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.mobile.MobileAuditRequestsService;
import ru.yandex.webmaster3.storage.mobile.data.MobileAuditRequestInfo;
import ru.yandex.webmaster3.storage.mobile.data.MobileAuditRequestState;
import ru.yandex.webmaster3.storage.mobile.data.MobileAuditResult;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author avhaliullin
 */
@Description("Информация о заявке на проверку урла на мобилопригодность")
@ReadAction
@Category("mobile")
public class MobileAuditRequestInfoAction extends AbstractUserVerifiedHostAction<MobileAuditInfoRequest, MobileAuditInfoResponse> {
    private MobileAuditRequestsService mobileAuditRequestsService;

    @Override
    public MobileAuditInfoResponse process(MobileAuditInfoRequest request) {
        long leftRequests = mobileAuditRequestsService.getRequestsLeftForHostToday(request.getHostId());
        MobileAuditRequestInfo latestRequest = mobileAuditRequestsService.getRequest(request.getUserId(), request.getHostId(), request.getRequestId());
        if (latestRequest == null) {
            return new MobileAuditInfoResponse.RequestNotFoundResponse();
        }
        if (latestRequest.getState() == MobileAuditRequestState.TASK_FINISHED) {
            switch (latestRequest.getResult().getType()) {
                case OK:
                    return new MobileAuditInfoResponse.NormalResponse(
                            leftRequests,
                            latestRequest.getCreatedAt(),
                            latestRequest.getState(),
                            latestRequest.getUrl(),
                            latestRequest.getResolution(),
                            asInstanceOf(latestRequest.getResult(), MobileAuditResult.Success.class)
                    );
                case INTERNAL_ERROR:
                    return new MobileAuditInfoResponse.InternalErrorResponse();
                case FETCH_FAILED:
                    MobileAuditResult.FetchFailedError fetchFailedError = asInstanceOf(latestRequest.getResult(), MobileAuditResult.FetchFailedError.class);
                    return new MobileAuditInfoResponse.UrlFetchFailedResponse(fetchFailedError.getHttpCode());
                default:
                    throw new RuntimeException("Unknown result type " + latestRequest.getResult().getType());
            }
        } else {
            return new MobileAuditInfoResponse.NormalResponse(
                    leftRequests,
                    latestRequest.getCreatedAt(),
                    latestRequest.getState(),
                    latestRequest.getUrl(),
                    latestRequest.getResolution(),
                    null
            );
        }
    }

    private <T extends MobileAuditResult> T asInstanceOf(MobileAuditResult result, Class<T> clazz) {
        if (!(clazz.isAssignableFrom(result.getClass()))) {
            throw new WebmasterException("Expected " + clazz.getSimpleName() + ", but found " + result.getClass().getSimpleName(),
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null));
        }
        return (T) result;
    }

    @Required
    public void setMobileAuditRequestsService(MobileAuditRequestsService mobileAuditRequestsService) {
        this.mobileAuditRequestsService = mobileAuditRequestsService;
    }
}
