package ru.yandex.webmaster3.viewer.http.niche;

import java.util.List;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Category;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.searchquery.OrderDirection;
import ru.yandex.webmaster3.storage.niche.NicheQueriesCHDao;
import ru.yandex.webmaster3.storage.niche.QuerySource;
import ru.yandex.webmaster3.storage.searchquery.DeviceType;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.host.verification.UsersVerifiedHostRequest;

/**
 * ishalaru
 * 28.06.2021
 **/
@Slf4j
@Component("/niche/queries")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Category("niche")
@ReadAction
public class GetNicheQueriesAction extends AbstractUserVerifiedHostAction<GetNicheQueriesAction.Request, GetNicheQueriesAction.Response> {
    private final NicheQueriesCHDao nicheQueriesCHDao;

    @Override
    public Response process(Request request) {
        final List<Pair<String, Long>> queries = nicheQueriesCHDao.getQueries(
                request.getHostId(), request.dateFrom.toLocalDate(), request.dateTo.toLocalDate(), request.regionId, request.deviceType,
                request.source, request.orderBy, request.orderDirection, request.pSize, request.p * request.pSize);
        long count = nicheQueriesCHDao.countQueries(request.getHostId(), request.dateFrom.toLocalDate(), request.dateTo.toLocalDate(),
                request.regionId, request.deviceType, request.source);

        return new Response(count, queries.stream().map(e -> new QueryInfo(e.getLeft(), e.getRight())).collect(Collectors.toList()));
    }

    public static class Request extends UsersVerifiedHostRequest {
        @Setter(onMethod_ = {@Description("Номер страницы"), @RequestQueryProperty(required = true)})
        int p;
        @Setter(onMethod_ = {@Description("Кол-во позиций на 1 странице"), @RequestQueryProperty(required = true)})
        int pSize;
        @Setter(onMethod_ = {@Description("Начало периода"), @RequestQueryProperty(required = true)})
        DateTime dateFrom;
        @Setter(onMethod_ = {@Description("Конец периода"), @RequestQueryProperty(required = true)})
        DateTime dateTo;
        @Setter(onMethod_ = {@Description("Тип устройства"), @RequestQueryProperty(required = true)})
        DeviceType deviceType;
        @Setter(onMethod_ = {@Description("Источник query"), @RequestQueryProperty(required = true)})
        QuerySource source;
        @Setter(onMethod_ = {@Description("Сортировка по"), @RequestQueryProperty})
        NicheQueriesCHDao.SortField orderBy = NicheQueriesCHDao.SortField.COUNT;
        @Setter(onMethod_ = {@Description("Порядок сортировки"), @RequestQueryProperty})
        OrderDirection orderDirection = OrderDirection.DESC;
        @Setter(onMethod_ = {@Description("Регионы"), @RequestQueryProperty(required = false)})
        private Integer[] regionId;
    }


    @Value
    public static class Response implements ActionResponse, ActionResponse.NormalResponse {
        @Description("Общее количество запросовв")
        long totalCount;
        @Description("Список запросов")
        List<QueryInfo> query;
    }

    @Value
    public static class QueryInfo {
        String query;
        Long countQueries;
    }
}
