package ru.yandex.webmaster3.viewer.http.recommendations;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.concurrency.AsyncCtx;
import ru.yandex.webmaster3.core.concurrency.AsyncTask;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.metrika.counters.AnonymousCounterBinding;
import ru.yandex.webmaster3.core.metrika.counters.CounterBindingStateEnum;
import ru.yandex.webmaster3.core.turbo.model.statistics.TurboClicksStatistics;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.achievements.dao.AchievementsCHDao;
import ru.yandex.webmaster3.storage.achievements.model.AchievementInfo;
import ru.yandex.webmaster3.storage.achievements.model.AchievementTld;
import ru.yandex.webmaster3.storage.achievements.model.AchievementType;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.HostDataState;
import ru.yandex.webmaster3.storage.host.HostIndicatorsDiff;
import ru.yandex.webmaster3.storage.host.HostIndicatorsState;
import ru.yandex.webmaster3.storage.host.service.CommercialOwnersService;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.importanturls.ImportantUrlsService;
import ru.yandex.webmaster3.storage.metrika.MetrikaCrawlStateService;
import ru.yandex.webmaster3.storage.metrika.dao.MetrikaCounterBindingStateYDao;
import ru.yandex.webmaster3.storage.metrika.data.MetrikaDomainCrawlState;
import ru.yandex.webmaster3.storage.notifications.service.UserNotificationSettingsService;
import ru.yandex.webmaster3.storage.recommendedquery.dao.HostsWithOpenedRecommendedYDao;
import ru.yandex.webmaster3.storage.services.CompanyInfo;
import ru.yandex.webmaster3.storage.services.IsShopHostInfo;
import ru.yandex.webmaster3.storage.services.MarketRatingInfo;
import ru.yandex.webmaster3.storage.services.ServiceInfo;
import ru.yandex.webmaster3.storage.services.SiteServiceType;
import ru.yandex.webmaster3.storage.services.SiteServicesCHDao;
import ru.yandex.webmaster3.storage.services.UseDirectInfo;
import ru.yandex.webmaster3.storage.sitestructure.SiteStructureService;
import ru.yandex.webmaster3.storage.turbo.service.TurboClickStatisticsService;
import ru.yandex.webmaster3.storage.turbo.service.TurboSearchUrlsStatisticsService;


@Service
@AllArgsConstructor(onConstructor_ = @Autowired)
public class WidgetInfoService {
    private static final AchievementTld DEFAULT_TLD = AchievementTld.RU;

    private final AbtService abtService;
    private final AchievementsCHDao mdbAchievementsCHDao;
    private final CommercialOwnersService commercialOwnersService;
    private final HostOwnerService hostOwnerService;
    private final HostsWithOpenedRecommendedYDao hostsWithOpenedRecommendedYDao;
    private final ImportantUrlsService importantUrlsService;
    private final MetrikaCounterBindingStateYDao metrikaCounterBindingStateYDao;
    private final MetrikaCrawlStateService metrikaCrawlStateService;
    private final MirrorService2 mirrorService2;
    private final SiteProblemsService siteProblemsService;
    private final SiteServicesCHDao mdbSiteServicesCHDao;
    private final SiteStructureService siteStructureService;
    private final TurboClickStatisticsService turboClickStatisticsService;
    private final TurboSearchUrlsStatisticsService turboSearchUrlsStatisticsService;
    private final UserNotificationSettingsService userNotificationSettingsService;

    protected Pair<List<Recommendation>, List<Widget>> getWidgetsAndRecommendations(AsyncCtx ctx,
                                                                                    WebmasterHostId hostId,
                                                                                    long userId) {
        @NotNull WebmasterHostId notNullOwner = hostOwnerService.getHostOwner(hostId);
        @NotNull String domain = WwwUtil.cutWWWAndM(hostId.getPunycodeHostname());

        List<Recommendation> recommendations = Collections.synchronizedList(new ArrayList<>());
        List<Widget> widgets = Collections.synchronizedList(new ArrayList<>());
        List<AsyncTask> asyncTasks = new ArrayList<>();

        recommendations.add(new Recommendation(RecommendationType.SITE_CONTENT_DATA));
        recommendations.add(new Recommendation(RecommendationType.YANDEX_PAY));
        recommendations.add(new Recommendation(RecommendationType.AUTHORIZATION));

        if (false) { // TODO
            recommendations.add(new Recommendation(RecommendationType.DIRECT));
        }

        asyncTasks.add(ctx.fork(() -> {
                    Pair<WebmasterHostId, DateTime> openedHostInfo = hostsWithOpenedRecommendedYDao.getOpenDate(hostId);

                    if (openedHostInfo == null) {
                        recommendations.add(new Recommendation(RecommendationType.RECOMMENDED_SEARCH_QUERY));
                    }
                }
        ));

        asyncTasks.add(ctx.fork(() -> {
                    Map<AchievementType, AchievementInfo> map = mdbAchievementsCHDao
                            .getAll(DEFAULT_TLD, domain)
                            .getInfos()
                            .stream()
                            .collect(W3Collectors.toEnumMap(AchievementInfo::getType, Function.identity(),
                                    AchievementType.class));

                    int speedGrade = 0;

                    if (map.containsKey(AchievementType.MOBILE_SPEED)) {
                        var info = map.get(AchievementType.MOBILE_SPEED);
                        if (info instanceof AchievementInfo.MobileSpeedAchievementInfo) {
                            speedGrade = ((AchievementInfo.MobileSpeedAchievementInfo) info).getValue();
                        }
                    }
                    if (speedGrade > 0 && abtService.isInExperiment(userId, Experiment.SPEED_GRADE_ACHIEVEMENT)) {
                        widgets.add(new SpeedGradeWidget(speedGrade));
                    }
                }
        ));

        asyncTasks.add(ctx.fork(() -> {
                    Map<SiteServiceType, ServiceInfo> siteServices = mdbSiteServicesCHDao.getSiteServiceInfo(domain);
                    MarketRatingInfo marketRatingInfo =
                            (MarketRatingInfo) siteServices.get(SiteServiceType.MARKET_RATING);
                    IsShopHostInfo isShop = (IsShopHostInfo) siteServices.get(SiteServiceType.IS_SHOP_HOST);
                    if (isShop != null && isShop.isValue()) {
                        recommendations.add(new Recommendation(RecommendationType.USLUGI));
                        if (marketRatingInfo == null || (marketRatingInfo.getId() == 0 && marketRatingInfo.getRating() == 0)) {
                            recommendations.add(new Recommendation(RecommendationType.MARKET));
                            recommendations.add(new Recommendation(RecommendationType.MARKETPLACE));
                        }
                    }

                    CompanyInfo companyInfo = (CompanyInfo) siteServices.get(SiteServiceType.COMPANY_INFO);
                    if (companyInfo == null) {
                        recommendations.add(new Recommendation(RecommendationType.SPRAVOCHNIK));
                    } else {
                        if (companyInfo.getUrlSocial() == null || companyInfo.getUrlSocial().isEmpty()) {
                            recommendations.add(new Recommendation(RecommendationType.SOCIAL_NETWORKS_SPRAVOCHNIK));
                        }
                    }
                }
        ));

        asyncTasks.add(ctx.fork(() -> {
                    List<Long> samples = metrikaCounterBindingStateYDao.getAllForDomain(domain)
                            .stream()
                            .filter(
                                    a -> a != null && a.getCounterBindingState() == CounterBindingStateEnum.APPROVED
                            )
                            .map(AnonymousCounterBinding::getCounterId)
                            .collect(Collectors.toList());
                    if (samples.isEmpty()) {
                        recommendations.add(new Recommendation(RecommendationType.METRIKA));
                    } else {
                        MetrikaDomainCrawlState domainCrawlState = metrikaCrawlStateService.getDomainCrawlState(domain);
                        if (domainCrawlState.getEnabledCounters().isEmpty()) {
                            recommendations.add(new Recommendation(RecommendationType.METRIKA_CRAWL));
                        }
                    }
                }
        ));

        asyncTasks.add(ctx.fork(() -> {
                    if (userNotificationSettingsService.getUserEmail(userId) == null) {
                        recommendations.add(new Recommendation(RecommendationType.NOTIFICATION_EMAIL));
                    }
                }
        ));

        asyncTasks.add(ctx.fork(() -> {
                    ImportantUrlsService.ImportantUrlsQuota quota =
                            importantUrlsService.getImportantUrlsQuota(hostId);
                    if (quota.getQuotaUsed() == 0) {
                        recommendations.add(new Recommendation(RecommendationType.MONITORING_IMPORTANT_PAGES));
                    } else {
                        long recommendationCount = quota.getQuotaRemain();

                        int searchableUrlsCount = importantUrlsService.getSearchableUrlsCount(hostId);
                        widgets.add(new MonitoringImportantPagesWidget(
                                quota.getQuotaUsed(),
                                Long.min(
                                        importantUrlsService.getRecommendedUrlsCount(hostId),
                                        recommendationCount
                                ),
                                quota.getQuotaTotal(),
                                Integer.max(quota.getQuotaUsed() - searchableUrlsCount, 0)
                        ));
                    }
                }
        ));

        asyncTasks.add(ctx.fork(() -> {
                    final TurboSearchUrlsStatisticsService.TurboDomainStats turboInfo =
                            turboSearchUrlsStatisticsService.getTurboInfo(domain);

            if (turboInfo.getTurboUrls() == 0) {
                recommendations.add(new Recommendation(RecommendationType.TURBO));
            } else {
                HostDataState dataState = siteStructureService.getHostDataState(hostId);
                TurboClicksStatistics clicksStats =
                        turboClickStatisticsService.turboClickStatistics(domain);
                long totalSearchPages = Optional.ofNullable(dataState)
                        .map(HostDataState::getHostIndicatorsDiff)
                        .map(HostIndicatorsDiff::getCurrentInfo)
                        .map(HostIndicatorsState::getSearchablePages)
                        .orElse(0L);

                if (turboInfo.getTurboSearchUrls() > 0 && clicksStats != null) {
                    widgets.add(new TurboPagesWidget(
                                    turboInfo.getTurboSearchUrls(),
                                    Long.max(totalSearchPages, turboInfo.getTurboSearchUrls()),
                                    clicksStats.getTotalClicks() == 0 ? 0 :
                                            (int) (clicksStats.getTurboClicks() * 100 / clicksStats.getTotalClicks())
                            )
                    );
                }
            }

            List<AbstractProblemInfo> problems = siteProblemsService.listProblemsForHost(hostId, null)
                            .stream()
                            .filter(p -> p.getUserVisibleState(mirrorService2.isMainMirror(hostId)) != SiteProblemState.NOT_APPLICABLE)
                            .filter(p -> p.getState() == SiteProblemState.PRESENT && p.getContent() != null).collect(Collectors.toList());

                    List<AbstractProblemInfo> nonTurboProblems =
                            problems.stream().filter(p -> !p.getProblemType().isTurboProblem()).collect(Collectors.toList());
                    if (!nonTurboProblems.isEmpty()) {
                        int recommendationsCount =
                                (int) nonTurboProblems.stream().filter(AbstractProblemInfo::isRecommendation).count();
                        widgets.add(new TechnicalProblemsWidget(recommendationsCount,
                                nonTurboProblems.size() - recommendationsCount));
                    }

                    List<AbstractProblemInfo> turboProblems = problems.stream()
                            .filter(p -> p.getProblemType().isTurboProblem()).collect(Collectors.toList());
                    List<AbstractProblemInfo> turboShopProblems = problems.stream()
                            .filter(p -> p.getProblemType().isTurboShopProblem(turboInfo.getTurboShopSearchUrls() > 0)).collect(Collectors.toList());
                    List<AbstractProblemInfo> turboContentProblems = problems.stream()
                            .filter(p -> p.getProblemType().isTurboContentProblem(turboInfo.getTurboContentSearchUrls() > 0)).collect(Collectors.toList());
                    if (!turboContentProblems.isEmpty()) {
                        int recommendationsCount =
                                (int) turboContentProblems.stream().filter(AbstractProblemInfo::isRecommendation).count();
                        widgets.add(new TurboProblemsWidget(recommendationsCount,
                                turboContentProblems.size() - recommendationsCount));
                    }
                    if (!turboShopProblems.isEmpty()) {
                        int recommendationsCount =
                                (int) turboShopProblems.stream().filter(AbstractProblemInfo::isRecommendation).count();
                        widgets.add(new ShopProblemsWidget(recommendationsCount,
                                turboShopProblems.size() - recommendationsCount));
                    }
                }
        ));
        asyncTasks.add(ctx.fork(() -> {
            ServiceInfo serviceInfo = mdbSiteServicesCHDao.getSiteServiceInfo(domain).get(SiteServiceType.USE_DIRECT);
            if (!(serviceInfo != null && ((UseDirectInfo) serviceInfo).isValue()) && commercialOwnersService.isCommercialOwner(notNullOwner)) {
                recommendations.add(new DirectWidget());
            }
        }));

        asyncTasks.forEach(AsyncTask::join);

        sortWidgets(widgets);
        sortRecommendations(recommendations);

        return Pair.of(recommendations, widgets);
    }

    public void fixEmailRecommendation(Long userId, List<Recommendation> recommendations) {
        boolean emailRec = userNotificationSettingsService.getUserEmail(userId) == null;
        boolean containsRec =
                recommendations.stream().anyMatch(a -> a.getType() == RecommendationType.NOTIFICATION_EMAIL);
        if (emailRec != containsRec) {
            if (emailRec) {
                recommendations.add(new Recommendation(RecommendationType.NOTIFICATION_EMAIL));
                sortRecommendations(recommendations);
            } else {
                recommendations.remove(new Recommendation(RecommendationType.NOTIFICATION_EMAIL));
            }
        }
    }

    public void sortRecommendations(List<Recommendation> recommendations) {
        recommendations.sort(Comparator.comparingInt(a -> a.getType().ordinal()));
    }

    public void sortWidgets(List<Widget> recommendations) {
        recommendations.sort(Comparator.comparingInt(a -> a.getType().ordinal()));
    }
}
