package ru.yandex.webmaster3.viewer.http.regions2;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.W3RegionInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.regions.RegionUtils;
import ru.yandex.webmaster3.core.regions.W3RegionsTreeService;
import ru.yandex.webmaster3.core.regions.data.HostRegion;
import ru.yandex.webmaster3.core.regions.data.HostRegionSourceTypeEnum;
import ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationRequest;
import ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationRequestStatus;
import ru.yandex.webmaster3.storage.host.moderation.regions.service.HostRegionsModerationService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author leonidrom
 */
@ReadAction
@Description("Получить информацию о регионах хоста")
@Category("regions")
public class HostRegionsInfoAction extends AbstractUserVerifiedHostAction<HostRegionsInfoRequest, HostRegionsInfoResponse> {
    private static final Logger log = LoggerFactory.getLogger(HostRegionsInfoAction.class);

    private HostRegionsModerationService hostRegionsModerationService;
    private W3RegionsTreeService w3regionsTreeService;

    @Override
    public HostRegionsInfoResponse process(HostRegionsInfoRequest request) {
        WebmasterHostId hostId = request.getHostId();

        HostRegionsModerationRequest moderationRequest = hostRegionsModerationService.getModerationRequest(hostId);
        HostRegionsModerationView moderationInfo = null;

        Pair<Set<Integer>, Set<Integer>> p = hostRegionsModerationService.getCurrentAndPendingModeratedRegions(hostId);
        Set<Integer> currentRegions = p.getLeft();
        Set<Integer> pendingRegions = p.getRight();

        if (moderationRequest != null && !moderationRequest.isUserClosed()) {
            HostRegionsModerationRequestStatus status = moderationRequest.getStatus();
            if (status == HostRegionsModerationRequestStatus.IN_MODERATION) {
                moderationInfo = new HostRegionsModerationView(HostRegionsModerationStatus.IN_MODERATION ,
                        moderationRequest.getRequestedRegions(), moderationRequest.getRequestId());
            } else if (status == HostRegionsModerationRequestStatus.MODERATED) {
                Set<Integer> requestedRegions = moderationRequest.getRequestedRegions();
                Set<Integer> acceptedRegions = moderationRequest.getAcceptedRegions();
                Set<Integer> rejectedRegions = Sets.difference(requestedRegions, acceptedRegions);

                if (!rejectedRegions.isEmpty() || !pendingRegions.isEmpty()) {
                    moderationInfo = new HostRegionsModerationView(HostRegionsModerationStatus.MODERATED,
                            acceptedRegions, rejectedRegions, moderationRequest.getRequestId());
                }
            }
        }

        Set<HostRegion> regions = hostRegionsModerationService.getHostRegionsInfo(hostId);

        List<Integer> dictionaryRegions = hostRegionsModerationService.filterRegions(regions, HostRegionSourceTypeEnum.DICTIONARY,
                RegionUtils.VISIBILITY_PREDICATE);

        int limit = hostRegionsModerationService.getHostRegionsLimit(hostId);
        if (limit < currentRegions.size()) {
            log.error("Invalid limits for {}: {} < {}", hostId.toStringId(), limit, currentRegions.size());
            limit = currentRegions.size();
        }

        return new HostRegionsInfoResponse.NormalResponse(limit,
                toRegionViews(dictionaryRegions),
                toRegionWithParentsViews(currentRegions),
                toRegionViews(pendingRegions.isEmpty() ? null : pendingRegions),
                moderationInfo);
    }

    private List<HostRegionsInfoResponse.RegionWithParentsView> toRegionWithParentsViews(Collection<Integer> regions) {
        return regions.stream()
                .map(this::toRegionWithParentsView)
                .collect(Collectors.toList());
    }

    private static List<RegionView> toRegionViews(Collection<Integer> regions) {
        if (regions == null) {
            return null;
        } else {
            return regions.stream()
                    .map(RegionView::new)
                    .collect(Collectors.toList());
        }
    }

    private HostRegionsInfoResponse.RegionWithParentsView toRegionWithParentsView(int regionId) {
        List<W3RegionInfo> parents = w3regionsTreeService.getVisibleParentsChain(regionId,
                RegionUtils.PARENT_VISIBILITY_PREDICATE);

        List<RegionView> regionViewParents = parents.stream()
                .map(r -> new RegionView(r.getId()))
                .collect(Collectors.toList());

        return new HostRegionsInfoResponse.RegionWithParentsView(regionId, regionViewParents);
    }

    @Required
    public void setHostRegionsModerationService(HostRegionsModerationService hostRegionsModerationService) {
        this.hostRegionsModerationService = hostRegionsModerationService;
    }

    @Required
    public void setW3regionsTreeService(W3RegionsTreeService w3regionsTreeService) {
        this.w3regionsTreeService = w3regionsTreeService;
    }
}
