package ru.yandex.webmaster3.viewer.http.robotstxt;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.async.AsyncRequestTaskService;
import ru.yandex.webmaster3.storage.async.model.AsyncOperationRequestInfo;
import ru.yandex.webmaster3.storage.async.model.HttpAsyncRequestInfoRequestState;
import ru.yandex.webmaster3.storage.async.model.HttpAsyncRequestInfoResultType;
import ru.yandex.webmaster3.storage.robotstxt.model.AnalyzeRobotsTxtErrorType;
import ru.yandex.webmaster3.storage.robotstxt.model.AnalyzeRobotsTxtResponseType;
import ru.yandex.webmaster3.storage.robotstxt.model.AnalyzeRobotsTxtResult;
import ru.yandex.webmaster3.storage.robotstxt.model.RobotsTxtErrorTransfer;
import ru.yandex.webmaster3.viewer.http.BaseWebmaster3Action;
import ru.yandex.webmaster3.viewer.http.async.model.AsyncInfoRequest;
import ru.yandex.webmaster3.viewer.http.robotstxt.model.AnalyzeRobotsTxtInfoResponse;

/**
 * @author: ishalaru
 * DATE: 20.05.2019
 */
@ReadAction
@Description(value = "Информация о заявке на анализ robots.txt")
@Category("robotstxt")
@Slf4j
@Component("/robotstxt/analyzeInfo")
public class AnalyzeRobotsTxtInfoAction extends BaseWebmaster3Action<AsyncInfoRequest, AnalyzeRobotsTxtInfoResponse> {
    private static final Duration ANSWER_WAITING_TIMEOUT = Duration.standardMinutes(1);

    private final AsyncRequestTaskService asyncRequestTaskService;

    @Autowired
    public AnalyzeRobotsTxtInfoAction(AsyncRequestTaskService asyncRequestTaskService) {
        this.asyncRequestTaskService = asyncRequestTaskService;
    }

    @Override
    public AnalyzeRobotsTxtInfoResponse process(AsyncInfoRequest request) {
        AsyncOperationRequestInfo responseInfo = asyncRequestTaskService.getResponseInfo(request.getRequestId());
        if (responseInfo == null) {
            return new AnalyzeRobotsTxtInfoResponse.RequestIdNotFound(this.getClass());
        }
        if (responseInfo.getState() == HttpAsyncRequestInfoRequestState.NEW &&
                responseInfo.getLastUpdate().plus(ANSWER_WAITING_TIMEOUT).isBefore(DateTime.now())) {
            log.error("Worker failed to start request in {}", ANSWER_WAITING_TIMEOUT);
            throw new WebmasterException("TimeoutError", new WebmasterErrorResponse.AsyncZoraErrorResponse(getClass()));
//            return new AnalyzeRobotsTxtInfoResponse.RequestFailed(this.getClass());
        }
        if (responseInfo.getState() != HttpAsyncRequestInfoRequestState.TASK_FINISHED) {
            return AnalyzeRobotsTxtInfoResponse.NormalResponse.stillInProgress();
//            return new AnalyzeRobotsTxtInfoResponse.RequestStillInProgress(this.getClass());
        }
        if (responseInfo.getResult() != HttpAsyncRequestInfoResultType.OK) {
            if (responseInfo.getResult() == HttpAsyncRequestInfoResultType.INTERNAL_ERROR) {
                throw new WebmasterException("InternalError", new WebmasterErrorResponse.AsyncZoraErrorResponse(getClass()));
            } else {
                return new AnalyzeRobotsTxtInfoResponse.RequestFailed(this.getClass());
            }
        }
        if (responseInfo.getResponseObjectTypeId() == AnalyzeRobotsTxtResponseType.NORMAL.getValue()) {
            AnalyzeRobotsTxtResult result = convertData(responseInfo, AnalyzeRobotsTxtResult.class);
            if (result == null) {
                return new AnalyzeRobotsTxtInfoResponse.ResponseDataCorrupted(this.getClass());
            }
            return makeResponse(result);
        } else if (responseInfo.getResponseObjectTypeId() == AnalyzeRobotsTxtResponseType.HTTP_PROBLEM.getValue()) {
            RobotsTxtErrorTransfer result = convertData(responseInfo, RobotsTxtErrorTransfer.class);
            return new AnalyzeRobotsTxtInfoResponse.AnalyzeRobotsTxtInfoHttpCodeResponse(this.getClass(),
                    result.getAnalyzeRobotsTxtErrorType(), result.getHttpResponsePart());

        } else {
            AnalyzeRobotsTxtErrorType analyzeRobotsTxtErrorType = convertData(responseInfo,
                    AnalyzeRobotsTxtErrorType.class);
            return new AnalyzeRobotsTxtInfoResponse.AnalyzeRobotsTxtInfoErrorResponse(this.getClass(),
                    analyzeRobotsTxtErrorType);

        }


    }

    protected <WorkClass> WorkClass convertData(AsyncOperationRequestInfo info, Class<WorkClass> clazz) {
        WorkClass result = null;
        try {
            result = JsonMapping.readValue(info.getResponse(), clazz);
        } catch (Exception e) {
            log.error("Problem with parsing response.", e);
        }
        return result;
    }

    public AnalyzeRobotsTxtInfoResponse makeResponse(AnalyzeRobotsTxtResult value) {
        return new AnalyzeRobotsTxtInfoResponse.NormalResponse(value.getRobotsTxtContent(),
                value.getParseErrors(), value.getAcceptedLines(), value.getAllowInfos(), false);
    }
}
