package ru.yandex.webmaster3.viewer.http.robotstxt;

import java.net.URL;
import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.UrlUtils;
import ru.yandex.webmaster3.storage.async.AsyncRequestTaskService;
import ru.yandex.webmaster3.storage.async.model.AsyncTaskType;
import ru.yandex.webmaster3.viewer.http.BaseWebmaster3Action;
import ru.yandex.webmaster3.viewer.http.robotstxt.model.StartAnalyzeRobotsTxtResponse;

/**
 * @author: ishalaru
 * DATE: 20.05.2019
 */
@ReadAction
@Description(value = "Заявка на анализ файла robots.txt")
@Category("robotstxt")
@Component("/robotstxt/startAnalyze")
@Slf4j
public class StartAnalyzeRobotsTxtAction extends BaseWebmaster3Action<AnalyzeRobotsTxtRequest, StartAnalyzeRobotsTxtResponse> {

    private final AsyncRequestTaskService asyncRequestTaskService;

    @Autowired
    public StartAnalyzeRobotsTxtAction(AsyncRequestTaskService asyncRequestTaskService) {
        this.asyncRequestTaskService = asyncRequestTaskService;
    }

    @Override
    public StartAnalyzeRobotsTxtResponse process(AnalyzeRobotsTxtRequest request) {
        UUID taskId = UUIDs.timeBased();
        validation(request.getHostName(), request.getRobotsTxtContent());
        AsyncRequestTaskService.CreateRequestResult requestResult =
                asyncRequestTaskService.createRequest(AsyncTaskType.ANALYZE_ROBOT_TXT, 1, request.create(), request.create(taskId));
        return new StartAnalyzeRobotsTxtResponse.NormalResponse(requestResult.getRequestId());
    }

    private void validation(String hostName, String robotsTxtContent) {
        URL hostUrl = null;
        if (StringUtils.isEmpty(hostName)) {
            if (StringUtils.isEmpty(robotsTxtContent)) {
                throw new WebmasterException("Url and robots txt are missing",
                        new WebmasterErrorResponse.IllegalParameterValueResponse(this.getClass(), "hostName", null));
            }
        } else {
            try {
                hostUrl = UrlUtils.prepareUrl(hostName, true);
            } catch (IllegalArgumentException e) {
                log.error("Unable to parse hostname: {}", hostName, e);
                throw new WebmasterException("Url and robots txt are missing",
                        new WebmasterErrorResponse.IllegalParameterValueResponse(this.getClass(), "hostName", hostName));
            }
        }
    }

}
