package ru.yandex.webmaster3.viewer.http.search;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.concurrency.AsyncCtx;
import ru.yandex.webmaster3.core.concurrency.AsyncTask;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.turbo.model.statistics.TurboClicksStatistics;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.achievements.dao.AchievementsCHDao;
import ru.yandex.webmaster3.storage.achievements.model.AchievementInfo;
import ru.yandex.webmaster3.storage.achievements.model.AchievementTld;
import ru.yandex.webmaster3.storage.achievements.model.AchievementType;
import ru.yandex.webmaster3.storage.checklist.data.SummarySiteProblemsInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.greenurls.CurrentSerpDataCHDao;
import ru.yandex.webmaster3.storage.serplinks.W3SerpLinksService;
import ru.yandex.webmaster3.storage.services.ServiceInfo;
import ru.yandex.webmaster3.storage.services.SiteServiceType;
import ru.yandex.webmaster3.storage.services.SiteServicesCHDao;
import ru.yandex.webmaster3.storage.turbo.dao.commerce.TurboMarketFeedsYDao;
import ru.yandex.webmaster3.storage.turbo.service.TurboClickStatisticsService;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.storage.turbo.service.TurboSearchUrlsStatisticsService;
import ru.yandex.webmaster3.storage.turbo.service.settings.TurboSettingsService;
import ru.yandex.webmaster3.viewer.http.concurrency.AsyncAction;
import ru.yandex.webmaster3.viewer.http.search.GetSearchSiteDisplayResponse.TurboYmlAlert;
import ru.yandex.webmaster3.viewer.http.search.GetSearchSiteDisplayResponse.YmlSnippetStatus;

/**
 * Created by shabashoff.
 */
@Slf4j
@ReadAction
@Category("search")
@Component("/search/display")
@AllArgsConstructor(onConstructor_ = @Autowired)
@Description("Получение инфы про представление сайта в поиске")
public class GetSearchSiteDisplayAction extends AsyncAction<GetSearchSiteDisplayRequest, GetSearchSiteDisplayResponse> {

    private final AbtService abtService;
    private final AchievementsCHDao mdbAchievementsCHDao;
    private final HostOwnerService hostOwnerService;
    private final SiteProblemsService siteProblemsService;
    private final SiteServicesCHDao mdbSiteServicesCHDao;
    private final TurboClickStatisticsService turboClickStatisticsService;
    private final FeedsNative2YDao feedsNativeYDao;
    private final TurboFeedsService turboFeedsService;
    private final TurboMarketFeedsYDao turboMarketFeedsYDao;
    private final TurboSearchUrlsStatisticsService turboSearchUrlsStatisticsService;
    private final TurboSettingsService turboSettingsService;
    private final W3SerpLinksService w3SerpLinksService;
    private final CurrentSerpDataCHDao currentSerpDataCHDao;

    @Override
    protected GetSearchSiteDisplayResponse processAsync(AsyncCtx ctx, GetSearchSiteDisplayRequest request) {
        @NotNull WebmasterHostId notNullOwner =
                Optional.of(hostOwnerService.getHostOwner(request.getHostId())).orElse(request.getHostId());
        @NotNull String domainOwner = WwwUtil.cutWWWAndM(notNullOwner);
        @NotNull String domain = WwwUtil.cutWWWAndM(request.getHostId());
        @NotNull GetSearchSiteDisplayResponse resp = new GetSearchSiteDisplayResponse();

        AsyncTask achFork = ctx.fork(() ->
                resp.setAchievements(
                        mdbAchievementsCHDao.getAchievementsAndProblems(AchievementTld.RU, domain)
                                .getInfos()
                                .stream()
                                .collect(W3Collectors.toEnumMap(AchievementInfo::getType, Function.identity(),
                                        AchievementType.class))
                )
        );

        AsyncTask alertsFork = ctx.fork(() -> {
            SummarySiteProblemsInfo problems = siteProblemsService.getRealTimeProblemsSummary(request.getHostId());
            TurboHostSettings settings = turboSettingsService.getSettings(domainOwner);
            List<TurboFeedSettings> feeds = turboFeedsService.getFeeds(domainOwner);

            resp.setFaviconAlert(
                    problems.isPresent(SiteProblemTypeEnum.FAVICON_ERROR)
                            || problems.isPresent(SiteProblemTypeEnum.MISSING_FAVICON)
            );
            resp.setTitleAlert(problems.isPresent(SiteProblemTypeEnum.DOCUMENTS_MISSING_TITLE));
            resp.setDescriptionAlert(problems.isPresent(SiteProblemTypeEnum.DOCUMENTS_MISSING_DESCRIPTION));

            if (feeds.stream().anyMatch(a -> a.isActive() && TurboFeedType.YML == a.getType())) {
                if (settings.getCommerceSettings().isOrderMethodSelected()) {
                    resp.setTurboYmlAlert(TurboYmlAlert.OK);
                } else {
                    resp.setTurboYmlAlert(TurboYmlAlert.CART_DISABLED);
                }
            }

            if (feeds.stream().anyMatch(a -> a.isActive() && TurboFeedType.RSS == a.getType())) {
                resp.setTurboRssAlert(GetSearchSiteDisplayResponse.TurboRssAlert.OK);
            } else {
                resp.setTurboRssAlert(GetSearchSiteDisplayResponse.TurboRssAlert.RSS_OFF);
            }

            if (problems.isPresent(SiteProblemTypeEnum.TURBO_LISTING_ERROR)) {
                resp.setTurboListingAlert(GetSearchSiteDisplayResponse.TurboListingAlert.DIAGNOSTIC_ALERT);
            } else if (resp.getTurboYmlAlert() == TurboYmlAlert.OK) {
                resp.setTurboListingAlert(GetSearchSiteDisplayResponse.TurboListingAlert.OK);
            }

            try {
                Pair<Long, Long> cnts = currentSerpDataCHDao.getGreenUrlsAndQAUrlsCount(request.getHostId());
                Long greenUrlsCnt = cnts.getLeft();
                Long qaUrlsCnt = cnts.getRight();
                log.info("Greens: {}, QA: {}", greenUrlsCnt, qaUrlsCnt);
                boolean greenUrlsActivated = greenUrlsCnt >= 1;
                boolean QAActivated = qaUrlsCnt >= 1;
                resp.setGreenUrls(!greenUrlsActivated);
                resp.setGreenUrlsAlert(!greenUrlsActivated);
                resp.setQaAlert(!QAActivated);
            } catch (Exception exp) {
                log.error(exp.getMessage(), exp);
            }
        });

        AsyncTask companyFork = ctx.fork(() -> {
                    final Map<SiteServiceType, ServiceInfo> siteServiceInfo =
                            mdbSiteServicesCHDao.getSiteServiceInfo(domain);
                    resp.setCompanyAlert(!siteServiceInfo.containsKey(SiteServiceType.COMPANY_INFO));
                    if (siteServiceInfo.containsKey(SiteServiceType.SNIPPET_PIC)) {
                        resp.setPicAlert(false);
                    }
                    if (siteServiceInfo.containsKey(SiteServiceType.IS_NEWS)) {
                        resp.setNewsAlert(false);
                    }
                }
        );

        AsyncTask turboFork = ctx.fork(() -> {
            resp.setTurboAlert(turboSearchUrlsStatisticsService.countTurboPages(domain) <= 0);
        });

        AsyncTask turboStatsFork = ctx.fork(() -> {

            TurboClicksStatistics clicksStats = turboClickStatisticsService.turboClickStatistics(domain);

            resp.setTurboTraffic(
                    clicksStats == null || clicksStats.getTotalClicks() == 0 ? 0 :
                            (int) (clicksStats.getTurboClicks() * 100 / clicksStats.getTotalClicks())
            );
        });

        AsyncTask serpLinks = ctx.fork(() -> {
            try {
                final List<String> sections = w3SerpLinksService.getSections(request.getHostId());
                resp.setSerpLink(sections.isEmpty()); // это на самом деле по логике serpLinkAlert
            } catch (Exception exp) {
                log.error(exp.getMessage(), exp);
            }
        });


        AsyncTask<Boolean> ymlSnippetFork =
                ctx.fork(() -> feedsNativeYDao.hasSuccessfulFeeds(domain) || turboMarketFeedsYDao.hasActiveFeeds(domain));

        achFork.join();
        alertsFork.join();
        turboFork.join();
        companyFork.join();
        turboStatsFork.join();
        serpLinks.join();
        TurboYmlAlert ymlStatus = resp.getTurboYmlAlert();
        if (ymlSnippetFork.join()) {
            resp.setYmlSnippetStatus(YmlSnippetStatus.ON);
        } else if (ymlStatus != TurboYmlAlert.NONE) {
            resp.setYmlSnippetStatus(YmlSnippetStatus.OFF);
        }

        return resp;
    }
}
