package ru.yandex.webmaster3.viewer.http.searchquery;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.*;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.searchquery.Query;
import ru.yandex.webmaster3.core.searchquery.QueryGroupId;
import ru.yandex.webmaster3.core.searchquery.QueryUtils;
import ru.yandex.webmaster3.core.searchquery.SpecialGroup;
import ru.yandex.webmaster3.storage.searchquery.QueryGroupService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

/**
 * @author tsyplyaev
 */
@WriteAction
@Category("searchquery")
@Component("/searchquery/group/uploaded/add")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class AddUploadedQueriesAction extends AbstractUserVerifiedHostAction<AddUploadedQueriesAction.Request, AddUploadedQueriesAction.Response> {
    private final QueryGroupService queryGroupService;

    @Override
    public Response process(Request request) {
        Set<String> queries;

        if (request.getQueries() == null) {
            if (request.getQueriesFile() != null) {
                if (request.getQueriesFile().getContentType().equals("application/octet-stream")) {
                    return new Response.IllegalFileFormatResponse();
                }

                Charset charset = Charset.forName("UTF-8");
                queries = new HashSet<>();
                try (Reader contentReader = new InputStreamReader(request.getQueriesFile().getInputStream(), charset)) {
                    BufferedReader br = new BufferedReader(contentReader);
                    String line;
                    while ((line = br.readLine()) != null) {
                        if (line.length() <= Query.MAX_LENGTH) {
                            String q = QueryUtils.CheckAndFixQueryStringUTF8(line);
                            if (!q.isEmpty()) {
                                queries.add(q);
                            }
                        }
                    }

                } catch (IOException e) {
                    throw new WebmasterException("IO error occured while reading request body",
                            new WebmasterErrorResponse.IllegalFileParameterResponse(SaveQueryGroupAction.class, "queriesFile", "Failed to read file content"), e);
                }
            } else {
                throw new WebmasterException("No queries or queriesFile parameter specified",
                        new WebmasterErrorResponse.IllegalFileParameterResponse(SaveQueryGroupAction.class, "queries", "No queries or queriesFile parameter specified"));
            }
        } else {
            queries = new HashSet<>(QueryUtils.CheckAndFixQueryStringUTF8(request.getQueries()));
        }

        if (queries.isEmpty()) {
            return new Response.NormalResponse();
        }

        QueryGroupId uploadedGroupId = new QueryGroupId(request.getHostId(), SpecialGroup.UPLOADED_QUERIES);
        if (queryGroupService.isLimitReached(uploadedGroupId, queries.size())) {
            int queriesLeft = queryGroupService.getQueriesLeft(uploadedGroupId);
            return new Response.QueryLimitReachedResponse(SpecialGroup.UPLOADED_QUERIES.name(), queriesLeft);
        }
        queryGroupService.addQueriesToGroup(uploadedGroupId, queries, true);

        if (request.getGroupId() != null) {
            Optional<QueryGroupId> groupIdO = QueryGroupId.byGroupIdStr(request.getHostId(), request.getGroupId());
            if (!groupIdO.isPresent()) {
                return new Response.NoSuchGroupResponse();
            }

            if (queryGroupService.isLimitReached(groupIdO.get(), queries.size())) {
                int queriesLeft = queryGroupService.getQueriesLeft(groupIdO.get());
                return new Response.QueryLimitReachedResponse(request.getGroupId(), queriesLeft);
            }
            queryGroupService.addQueriesToGroup(groupIdO.get(), queries, false);
        }
        return new Response.NormalResponse();
    }

    public static class Request extends AbstractUserVerifiedHostRequest {
        private String[] queries;
        private FileParameter queriesFile;
        private String groupId = null;

        public String[] getQueries() {
            return queries;
        }

        @RequestPostProperty
        public void setQueries(String[] queries) {
            this.queries = queries;
        }

        public FileParameter getQueriesFile() {
            return queriesFile;
        }

        @RequestFileProperty
        public void setQueriesFile(FileParameter queriesFile) {
            this.queriesFile = queriesFile;
        }

        public String getGroupId() {
            return groupId;
        }

        @RequestQueryProperty
        public void setGroupId(String groupId) {
            this.groupId = groupId;
        }
    }

    public static abstract class Response implements ActionResponse {
        public static class NormalResponse extends Response implements ActionResponse.NormalResponse {
        }

        public static class NoSuchGroupResponse extends Response implements ActionResponse.ErrorResponse {
            @Override
            public Enum<?> getCode() {
                return ErrorType.SEARCHQUERIES__NO_SUCH_GROUP;
            }
        }

        public static class QueryLimitReachedResponse extends Response implements ActionResponse.ErrorResponse {
            private final String groupId;
            private final int queriesLeft;

            public QueryLimitReachedResponse(String groupId, int queriesLeft) {
                this.groupId = groupId;
                this.queriesLeft = queriesLeft;
            }

            public String getGroupId() {
                return groupId;
            }

            public int getQueriesLeft() {
                return queriesLeft;
            }

            @Override
            public Enum<?> getCode() {
                return ErrorType.SEARCHQUERIES__QUERY_LIMIT_REACHED;
            }
        }

        public static class IllegalFileFormatResponse extends Response implements ActionResponse.ErrorResponse {
            @Override
            public Enum<?> getCode() {
                return ErrorType.SEARCHQUERIES__ILLEGAL_FILE_FORMAT;
            }
        }
    }
}
