package ru.yandex.webmaster3.viewer.http.searchquery;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.searchquery.LatestSearchQuery;
import ru.yandex.webmaster3.core.searchquery.QueryFilter;
import ru.yandex.webmaster3.core.searchquery.QueryGroup;
import ru.yandex.webmaster3.core.searchquery.QueryGroupId;
import ru.yandex.webmaster3.core.searchquery.QueryId;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.core.searchquery.SpecialGroup;
import ru.yandex.webmaster3.core.searchquery.viewer.QueryFilterConverter;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.storage.searchquery.LatestQueriesService;
import ru.yandex.webmaster3.storage.searchquery.QueryGroupService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author aherman
 */
@ReadAction
@Category("searchquery")
@Component("/searchquery/latest/filter")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class GetLatestQueriesAction extends AbstractUserVerifiedHostAction<GetLatestQueriesRequest, GetLatestQueriesResponse> {

    private final QueryGroupService queryGroupService;
    private final LatestQueriesService latestQueriesService;

    @Override
    public GetLatestQueriesResponse process(GetLatestQueriesRequest request) {
        WebmasterHostId hostId = request.getHostId();
        List<QueryFilter> filters = QueryFilterConverter.toFilter(request.getFilters());

        Optional<QueryGroupId> groupIdO = QueryGroupId.byGroupIdStr(request.getHostId(), request.getGroupId());
        if (filters.isEmpty()) {
            if (!groupIdO.isPresent()) {
                return new GetLatestQueriesResponse.NoSuchGroupResponse();
            }
            QueryGroupId groupId = groupIdO.get();
            if (groupId.isSpecial()) {
                if (groupId.getSpecialGroup() != SpecialGroup.ALL_QUERIES) {
                    return new GetLatestQueriesResponse.NoSuchGroupResponse();
                }
            } else {
                QueryGroup group = queryGroupService.getGroup(groupId);
                if (group == null) {
                    return new GetLatestQueriesResponse.NoSuchGroupResponse();
                }
                if (group.getFilters() == null || group.getFilters().isEmpty()) {
                    return new GetLatestQueriesResponse.NormalResponse(0, Collections.emptyList());
                }
                filters = group.getFilters();
            }
        }

        QueryIndicator showIndicator;
        if (request.getShow() == GetLatestQueriesRequest.ShowIndicator.QUERY) {
            showIndicator = QueryIndicator.QUERY;
        } else {
            showIndicator = QueryIndicator.URL;
        }

        long totalQueries;

        List<GetLatestQueriesResponse.Query> result;
        if (showIndicator == QueryIndicator.QUERY) {
            totalQueries = latestQueriesService.countQueriesByFilter(hostId, filters);
            PageUtils.LongPager page = PageUtils.getPage(request.getPage(), request.getPageSize(), totalQueries);
            if (page.isEmpty()) {
                return new GetLatestQueriesResponse.NormalResponse(totalQueries, Collections.emptyList());
            }

            result = getQueries(hostId, filters, request, page.toRangeStart(), page.getPageSize());
        } else {
            totalQueries = latestQueriesService.countUrlsByFilter(hostId, filters);

            PageUtils.LongPager page = PageUtils.getPage(request.getPage(), request.getPageSize(), totalQueries);
            if (page.isEmpty()) {
                return new GetLatestQueriesResponse.NormalResponse(totalQueries, Collections.emptyList());
            }
            result = getUrls(hostId, filters, request, page.toRangeStart(), page.getPageSize());
        }
        return new GetLatestQueriesResponse.NormalResponse(totalQueries, result);
    }

    private List<GetLatestQueriesResponse.Query> getQueries(WebmasterHostId hostId,
                                                            List<QueryFilter> filters, GetLatestQueriesRequest request, long from, long limit) {
        List<LatestSearchQuery> queries;
        queries = latestQueriesService.listQueriesByFilter(hostId, filters, from, limit,
                request.getOrderBy(), request.getDirection());
        Map<QueryId, String> queryTexts;
        List<QueryId> ids = queries.stream().map(LatestSearchQuery::getQueryId).collect(Collectors.toList());
        queryTexts = latestQueriesService.getQueryTextsByIds(request.getHostId(), ids);

        return GetLatestQueriesResponse.toQueries(queries, queryTexts);
    }

    private List<GetLatestQueriesResponse.Query> getUrls(WebmasterHostId hostId,
                                                         List<QueryFilter> filters, GetLatestQueriesRequest request, long from, long limit) {
        List<LatestSearchQuery> queries;
        queries = latestQueriesService.getUrlsByFilter(hostId, filters, from, limit, request.getOrderBy(), request.getDirection());

        List<GetLatestQueriesResponse.Query> result = new ArrayList<>();
        for (LatestSearchQuery query : queries) {
            Map<QueryIndicator, Object> statistics = new HashMap<>();
            statistics.put(QueryIndicator.URL, query.getUrl());
            statistics.put(QueryIndicator.TOTAL_SHOWS_COUNT, query.getTotalShowsCount());
            statistics.put(QueryIndicator.TOTAL_CLICKS_COUNT, query.getTotalClicksCount());
            statistics.put(QueryIndicator.TOTAL_CTR, query.getCtrTotal());
            statistics.put(QueryIndicator.AVERAGE_SHOW_POSITION, query.getAvgShowPosition());
            statistics.put(QueryIndicator.AVERAGE_CLICK_POSITION, query.getAvgClickosition());
            GetLatestQueriesResponse.Query q = new GetLatestQueriesResponse.Query(statistics);
            result.add(q);
        }

        return result;
    }
}
